/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package b1module;
import java.util.Random;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import javax.crypto.Cipher;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import javax.swing.*;
import java.awt.event.*;
/**
 *
 * Created by Marcin Baliniak (Eccel Technology Ltd).
 * Release Date: 25/05/2017
 * Version : 1.0
 */


public class B1Module {

    /// <summary>
    /// Enumeration representing UART commands.
    /// </summary>
    
    public class TagType {
        public static final byte NoTag = 0x00;
        public static final byte NotComplete = 0x01;
        public static final byte Ultralight = 0x02;
        public static final byte UltralightEV1_80B = 0x03;
        public static final byte UltralightEV1_164B = 0x04;
        public static final byte ClassicMini = 0x05;
        public static final byte Classic1K  = 0x06;
        public static final byte Classic4K = 0x07;
        public static final byte NTAG203F = 0x08;
        public static final byte NTAG210 = 0x09;
        public static final byte NTAG212 = 0x0A;
        public static final byte NTAG213F = 0x0B;
        public static final byte NTAG216F = 0x0C;
        public static final byte NTAG213 = 0x0D;
        public static final byte NTAG215 = 0x0E;
        public static final byte NTAG216 = 0x0F;
        public static final byte Unknown =0x10;        
    }
    
    public class UartCommandT {
        public static final byte Dummy = 0x00;
        public static final byte WriteToRfidMemory = 0x01;
        public static final byte ReadFromRfidMemory = (0x02);
        public static final byte EnterSleepMode = (0x03);
        public static final byte Reset = 0x04;
        public static final byte SetBaudRate = 0x05;
        public static final byte SetDataType = 0x06;
        public static final byte SetHeaderType = 0x07;
        public static final byte SetIoState = 0x08;
        public static final byte ReadIoState = 0x09;
        public static final byte SetIoInterrupt = 0x0A;
        public static final byte MeasureVoltage = 0x0B;
        public static final byte MeasureDieTemperature = 0x0C;
        public static final byte SetIdacCurrent = 0x0D;
        public static final byte EnableComparator = 0x0E;
        public static final byte DisableComparator = 0x0F;
        public static final byte EnablePWM = 0x10;
        public static final byte SetAesInitVector = 0x11;
        public static final byte SetAesKey = 0x12;
        public static final byte ReadAesInitVector = 0x13;
        public static final byte ReadAesKey = 0x14;
    }

    /// <summary>
    /// Enumeration respresenting the possible responses from B1 module as described in User Manual. Last two 0xFE and 0xFF are not directly send from B1 module.
    /// They are generated by this class when invalid packet is received or when timeout is generater per the specification.
    /// </summary>
    public class ResponseT {
        public static final byte ACK = 0;
        public static final byte InvalidCommand = 0x01;
        public static final byte InvalidParameter = 0x02;
        public static final byte ProtocolError = 0x03;
        public static final byte MemoryError = 0x04;
        public static final byte SystemError = 0x05;
        public static final byte ModuleTimeout = 0x06;
        public static final byte Overflow = 0x07;
        public static final byte AsyncPacket = 0x08;
        public static final byte Busy = 0x09;
        public static final byte SystemStart = 0x0A;
        
        public static final byte InvalidPacket = (byte)0xFE;
        public static final byte ClientTimeout = (byte)0xFF;
    }
    
    /// <summary>
    /// Class used as argument when Response event is raised.
    /// </summary>
    public class ResponseArgs {
        public byte Response;      
        public byte[] Parameters;
        public char HeaderCRC;
        public char DataCRC;
        
        public ResponseArgs(byte Response, byte[] Parameters, char HeaderCRC, char DataCRC) {
            this.Response   = Response;
            this.Parameters = Parameters;
            this.HeaderCRC  = HeaderCRC;
            this.DataCRC    = DataCRC;
        }
        
        public ResponseArgs() {
            this.Response   = 0x00;
            this.Parameters = null;
            this.HeaderCRC  = 0;
            this.DataCRC    = 0;
        }
    }
    
    /// <summary>
    /// Class used as argument when Request event is raised.
    /// </summary>
    public class RequestArgs {
        public byte[] Data;
        
        public RequestArgs(byte[] Data) {
            this.Data = Data;
        }
    } 
    
    /// <summary>
    /// Enumeration representing possible DataType configurations.
    /// </summary>
    public class DataTypeT {
        public static final byte Plain = 0;
        public static final byte Encrypted = 1;
    };
    /// <summary>
    /// Enumeratuon representing possible HeaderType configurations.
    /// </summary>
    public class HeaderTypeT {
        public static final byte A = 0;
        public static final byte B = 1;
    }
    /// <summary>
    /// Enumeration of the B1 IOs.
    /// </summary>
    public class GpioT {
        public static final byte Gpio0 = 0;
        public static final byte Gpio1 = 1;
        public static final byte Gpio2 = 2;
        public static final byte Gpio3 = 3;              
    }
    
    /// <summary>
    /// Enumeratuon representing possible GpioState configurations.
    /// </summary>
    public class GpioStateT {
        public static final byte DisabledHiZ = 0;
        public static final byte Input = 1;
        public static final byte OutputLow = 2;
        public static final byte OutputHigh = 3;  
    }
    
    /// <summary>
    /// Enumeratuon representing possible GpioInterrupt configurations.
    /// </summary>
    public class GpioInterruptT {
        public static final byte FallingEdge = 0;
        public static final byte RisingEdge = 1;
        public static final byte AnyEdge = 2;
        public static final byte Disabled = 3;         
    }
    
    /// <summary>
    /// Enumeratuon representing possible AdcSource configurations.
    /// </summary>
    public class AdcSourceT {
        public static final byte AdcPin = 0;
        public static final byte PowerSupply = 1;       
    }
    
    /// <summary>
    /// Enumeratuon representing possible VoltageFormat configurations.
    /// </summary>
    public class VoltageFormatT {
        public static final byte Uint32_mv = 0;
        public static final byte Float_mV = 1;
        public static final byte Float_V = 2;         
    }
    
    /// <summary>
    /// Enumeratuon representing possible CurrentFormat configurations.
    /// </summary>
    public class CurrentFormatT {
        public static final byte Int32_nA = 0;
        public static final byte Float_nA = 1;
        public static final byte Float_uA = 2;
    }
    
    /// <summary>
    /// Enumeratuon representing possible PWMConfigFormat configurations.
    /// </summary>
    public class PWMConfigFormatT {
        public static final byte Uint32_Hz = 0;
        public static final byte Float_Hz = 1;
        public static final byte Uint32_us = 2;
        public static final byte Float_s = 3;
    }
    
    /// <summary>
    /// Enumeratuon representing possible TemperatureFormat configurations.
    /// </summary>
    public class TemperatureFormatT {
        public static final byte Int32_mC = 0;
        public static final byte Float_mC = 1;
        public static final byte Float_C = 2;
        public static final byte Int32_mF = 3;
        public static final byte Float_mF = 4;
        public static final byte Float_F = 5;
    }
    
    /// <summary>
    /// Enumeratuon representing possible ComparatorReferenceVoltage configurations.
    /// </summary>
    public class ComparatorReferenceVoltageT {
        public static final byte Ref1V25 = 0;
        public static final byte Ref2V5 = 1;
        public static final byte RefDivVdd = 2;
    }
    
    /// <summary>
    /// Enumeratuon representing possible ComparatorOutputPin configurations.
    /// </summary>
    public class ComparatorOutputPinT {
        public static final byte Disabled = 0;
        public static final byte Enabled = 1;
        public static final byte EnabledNegated = 2;
    }
    
    /// <summary>
    /// Enumeratuon representing possible ComparatorAsyncPacketEdgeSensitivity configurations.
    /// </summary>
    public class ComparatorAsyncPacketEdgeSensitivityT {
        public static final byte Disabled = 0;
        public static final byte FallingEdge = 1;
        public static final byte RisingEdge = 2;
        public static final byte AnyEdge = 3;
    }  


        /// <summary>
        /// Table used when calculating CRC16.
        /// </summary>
    private final char[] CCITTCRCTable = {
        0x0000, 0x1021, 0x2042, 0x3063, 0x4084, 0x50a5, 
        0x60c6, 0x70e7, 0x8108, 0x9129, 0xa14a, 0xb16b, 
        0xc18c, 0xd1ad, 0xe1ce, 0xf1ef, 0x1231, 0x0210, 
        0x3273, 0x2252, 0x52b5, 0x4294, 0x72f7, 0x62d6, 
        0x9339, 0x8318, 0xb37b, 0xa35a, 0xd3bd, 0xc39c, 
        0xf3ff, 0xe3de, 0x2462, 0x3443, 0x0420, 0x1401, 
        0x64e6, 0x74c7, 0x44a4, 0x5485, 0xa56a, 0xb54b, 
        0x8528, 0x9509, 0xe5ee, 0xf5cf, 0xc5ac, 0xd58d, 
        0x3653, 0x2672, 0x1611, 0x0630, 0x76d7, 0x66f6, 
        0x5695, 0x46b4, 0xb75b, 0xa77a, 0x9719, 0x8738, 
        0xf7df, 0xe7fe, 0xd79d, 0xc7bc, 0x48c4, 0x58e5, 
        0x6886, 0x78a7, 0x0840, 0x1861, 0x2802, 0x3823, 
        0xc9cc, 0xd9ed, 0xe98e, 0xf9af, 0x8948, 0x9969, 
        0xa90a, 0xb92b, 0x5af5, 0x4ad4, 0x7ab7, 0x6a96, 
        0x1a71, 0x0a50, 0x3a33, 0x2a12, 0xdbfd, 0xcbdc, 
        0xfbbf, 0xeb9e, 0x9b79, 0x8b58, 0xbb3b, 0xab1a, 
        0x6ca6, 0x7c87, 0x4ce4, 0x5cc5, 0x2c22, 0x3c03, 
        0x0c60, 0x1c41, 0xedae, 0xfd8f, 0xcdec, 0xddcd, 
        0xad2a, 0xbd0b, 0x8d68, 0x9d49, 0x7e97, 0x6eb6, 
        0x5ed5, 0x4ef4, 0x3e13, 0x2e32, 0x1e51, 0x0e70, 
        0xff9f, 0xefbe, 0xdfdd, 0xcffc, 0xbf1b, 0xaf3a, 
        0x9f59, 0x8f78, 0x9188, 0x81a9, 0xb1ca, 0xa1eb, 
        0xd10c, 0xc12d, 0xf14e, 0xe16f, 0x1080, 0x00a1, 
        0x30c2, 0x20e3, 0x5004, 0x4025, 0x7046, 0x6067, 
        0x83b9, 0x9398, 0xa3fb, 0xb3da, 0xc33d, 0xd31c, 
        0xe37f, 0xf35e, 0x02b1, 0x1290, 0x22f3, 0x32d2, 
        0x4235, 0x5214, 0x6277, 0x7256, 0xb5ea, 0xa5cb, 
        0x95a8, 0x8589, 0xf56e, 0xe54f, 0xd52c, 0xc50d, 
        0x34e2, 0x24c3, 0x14a0, 0x0481, 0x7466, 0x6447, 
        0x5424, 0x4405, 0xa7db, 0xb7fa, 0x8799, 0x97b8, 
        0xe75f, 0xf77e, 0xc71d, 0xd73c, 0x26d3, 0x36f2, 
        0x0691, 0x16b0, 0x6657, 0x7676, 0x4615, 0x5634, 
        0xd94c, 0xc96d, 0xf90e, 0xe92f, 0x99c8, 0x89e9, 
        0xb98a, 0xa9ab, 0x5844, 0x4865, 0x7806, 0x6827, 
        0x18c0, 0x08e1, 0x3882, 0x28a3, 0xcb7d, 0xdb5c, 
        0xeb3f, 0xfb1e, 0x8bf9, 0x9bd8, 0xabbb, 0xbb9a, 
        0x4a75, 0x5a54, 0x6a37, 0x7a16, 0x0af1, 0x1ad0, 
        0x2ab3, 0x3a92, 0xfd2e, 0xed0f, 0xdd6c, 0xcd4d, 
        0xbdaa, 0xad8b, 0x9de8, 0x8dc9, 0x7c26, 0x6c07, 
        0x5c64, 0x4c45, 0x3ca2, 0x2c83, 0x1ce0, 0x0cc1, 
        0xef1f, 0xff3e, 0xcf5d, 0xdf7c, 0xaf9b, 0xbfba, 
        0x8fd9, 0x9ff8, 0x6e17, 0x7e36, 0x4e55, 0x5e74, 
        0x2e93, 0x3eb2, 0x0ed1, 0x1ef0 
    }; 

    private boolean Receiving = false;
    private final byte STX = 0x02;
    private final byte ETX = 0x03;
    private final byte DLE = 0x10;
    private int ByteCounter;
    private char RxDataSize;
    private byte[] RxBuffer = new byte[65536];
    private byte DataType = DataTypeT.Plain;
    private byte HeaderType = HeaderTypeT.A;
    private byte NewHeaderType = HeaderTypeT.A;    
    private byte[] AesKey = new byte[16];
    private byte[] AesIv = new byte[16];
    private boolean DLEReceived = false;
    private Timer TimeOutTimer;
    private final int TimeOutValue_ms = 100;
    private ResponseArgs ProcessedPacket;
    public final int TimeoutMs_Startup = (100);
    public final int TimeoutMs_Interrupt = (50);
    public final int TimeoutMs_Comparator = (50);
    public final int TimeoutMs_CommunicationParameterChange = (50);

    /**
     *Constructor sets up the AES Key and IV to zeroes and launches timer.
     */
    public B1Module() {
        AesKey = new byte[16];
        AesIv = new byte[16];

        for (int i = 0; i < 16; ++i) {
            AesKey[i] = 0;
            AesIv[i] = 0;
        }

        ActionListener TimeOutListener = new ActionListener(){
          @Override            
          public void actionPerformed(ActionEvent event){
            System.out.println("timeout!!!");
            Receiving = false;
            //GenerateClientTimeoutEvent();            
          }
        };       

        TimeOutTimer = new Timer(TimeOutValue_ms, TimeOutListener);
        TimeOutTimer.setRepeats(false);
        TimeOutTimer.setInitialDelay(TimeOutValue_ms);
    }   
    
    //calback - user should override this 
    public void Response(ResponseArgs args) 
    {
        throw new UnsupportedOperationException("Response not defined!");
    }

    public void Request(RequestArgs args) 
    {
        throw new UnsupportedOperationException("Request not defined!");
    } 

    /// <summary>
    /// Method used to raise Response event with Invalid Packet response type.
    /// </summary>
    private void GenerateInvalidPacketEvent() {
        Response(new ResponseArgs(ResponseT.InvalidPacket, null, (char)0, (char)0));
    }

    /// <summary>
    /// Method used to raise Response event with Client Timeout response type.
    /// </summary>
    private void GenerateClientTimeoutEvent() {
        Response(new ResponseArgs(ResponseT.ClientTimeout, null, (char)0, (char)0));
    }

    private boolean ResponseIsDefined(int type) {
        if (type > 0x0A && type < 0xfe)
               return false;
        return true;
    }        
        
    /// <summary> Private method which receives packed data and process it. </summary>
    /// <para> This method is called after receiving full packet with valid header. When the method is executed on a valid packet data it fills 
    /// the ProcessedPacket.DataCRC private field. Method recognizes object configuration for the packet data. Method doesn't check for the minimum size of the 
    /// packet data array as it is called from the SupplyRxData which checks for the size. </para>
    /// <param name="PacketData"> Array of bytes representing packet data after receiving packet with valid header. The size of it shall be minimum 3 bytes. </param>
    private void ProcessPacketData(byte[] PacketData) {
        if (DataType == DataTypeT.Plain) {

            ProcessedPacket.DataCRC = (char)((PacketData[PacketData.length - 1] << 8) | (PacketData[PacketData.length - 2] & 0x00ff));                

            if (ProcessedPacket.DataCRC != GetCRC(PacketData, 0, PacketData.length - 2)) {
                    // If the packet data CRC is not correct we are going to send invalid packet and finish execution of this command.
                GenerateInvalidPacketEvent();
                return;
            }

            if (ResponseIsDefined((int)(PacketData[0] & 0xff)) == false) {
                // If the response is not recognized we are going to send invalid packet and finish execution of this command.
                GenerateInvalidPacketEvent();
                return;
            }

            ProcessedPacket.Response = PacketData[0];

            if (PacketData.length > 3) {
                // When we have at least one byte of parameters then we have to create a new array for them.
                ProcessedPacket.Parameters = new byte[PacketData.length - 3];

                System.arraycopy(PacketData, 1, ProcessedPacket.Parameters, 0, ProcessedPacket.Parameters.length);
            } else {
                // There are no parameters bytes in the packet data.
                ProcessedPacket.Parameters = null;	                    
            }

            Response(ProcessedPacket);
        } else if (DataType == DataTypeT.Encrypted) {
                if (PacketData.length % 16 != 0) {
                    GenerateInvalidPacketEvent();
                    return;
                }

                byte[] PlainData = Decrypt(PacketData);

                // indexing is count from 1
                char LastValidIndex = (char)((PlainData[1] << 8) | (PlainData[0] & 0x00ff));                    

                if (LastValidIndex < 5) {
                    GenerateInvalidPacketEvent();
                    return;
                }

                ProcessedPacket.DataCRC = (char)((PlainData[LastValidIndex - 1] << 8) | (PlainData[LastValidIndex - 2] & 0x00ff));

                if (ProcessedPacket.DataCRC != GetCRC(PlainData, 0, LastValidIndex - 2)) {
                    GenerateInvalidPacketEvent();
                    return;
                }

                if (ResponseIsDefined(PlainData[2]) == false) {
                    GenerateInvalidPacketEvent();
                    return;
                }

                ProcessedPacket.Response = PlainData[2];

                if (LastValidIndex > 5) {
                    // When we have at least one byte of parameters then we have to create a new array for them.
                    ProcessedPacket.Parameters = new byte[LastValidIndex - 5];
                    System.arraycopy(PlainData, 3, ProcessedPacket.Parameters, 0, ProcessedPacket.Parameters.length);
                } else {
                    // There are no parameters bytes in the packet data.
                    ProcessedPacket.Parameters = null;                        
                }

                Response(ProcessedPacket);
        }
    }

    /// <summary> Method used by the client to supply to the object received data from the B1 / USB-B1 module. </summary>
    /// <para> When a packed is received inside this method the header is processed if the object is set up to Header Type A. Then packet data is extracted and 
    /// passed to private method ProcessPacketData. Private field ProcessedPacket.HeaderCRC is filled. If the object is configured as Header Type B then 
    /// ProcessedPacket.HeaderCRC is set to 0 and packet data is extracted and passed to the private method ProcessPacketData. 
    /// Method checks for the minimum size of the packet data and for the correct header. </para>
    /// <param name="Data"> Array of bytes received from the B1 / USB-B1 module. </param>
    /// <returns>  NotInitialised - the object is not yet initialized (it doesn't have response handler set).
    ///            InvalidParameter - array passed to the method is null
    ///            OK - data was processed.
    /// </returns>
    public void SupplyRxData(byte[] Data) {

        if (Data.length == 0)
            throw new UnsupportedOperationException("Data is null");


        for (byte DataByte : Data) {
            switch (HeaderType) {
                case (HeaderTypeT.A):
                if (Receiving == false) {
                    if (DataByte == STX) {
                        // Begining of the packet
                        // Reset timeout timer
                        TimeOutTimer.restart();
                        ProcessedPacket = new ResponseArgs();
                        RxBuffer[0]     = STX; // we are storing STX as the Header CRC is calculated with STX
                        Receiving       = true;
                        ByteCounter     = 1;
                        RxDataSize      = 0;
                    } else {
                            // Byte outside packet
                        // Stop timeout timer
                        TimeOutTimer.stop();
                    }
                } else {
                         // Reset timeout timer
                    TimeOutTimer.restart(); 
                    RxBuffer[ByteCounter++] = DataByte;

                    if (ByteCounter == 5) {
                        // We have received a full header
                        // PacketSize = STX + 2 bytes DataSize + 2 bytes header CRC + DataSize = ByteCounter   

                        ProcessedPacket.HeaderCRC = (char)((RxBuffer[4] << 8) | (RxBuffer[3] & 0x00ff));

                        if (ProcessedPacket.HeaderCRC != GetCRC(RxBuffer, 0, 3)) {
                            GenerateInvalidPacketEvent();
                            Receiving = false;
                            HeaderType = NewHeaderType;
                            return;
                        }

                        RxDataSize = (char)((RxBuffer[2] << 8) | (RxBuffer[1] & 0x00ff));
                         if (RxDataSize < 3) {
                            // Packet data has to be at least 3 bytes long - Command + 2 bytes CRC
                            // Stop timeout timer
                            TimeOutTimer.stop();
                            GenerateInvalidPacketEvent();
                            Receiving = false;                                    
                            HeaderType = NewHeaderType;
                            break;
                        }                                                                                               
                    } else if (ByteCounter == RxBuffer.length) {
                        // We have received too many bytes and exceeded RxBuffer size
                        // Stop timeout timer
                        TimeOutTimer.stop();
                        GenerateInvalidPacketEvent();
                        Receiving = false;
                        HeaderType = NewHeaderType;
                        break;
                        } else if (ByteCounter == (5 + RxDataSize)) {
                            // Stop timeout timer
                        TimeOutTimer.stop();
                        Receiving = false;
                        HeaderType = NewHeaderType;
                        byte[] PacketData = new byte[RxDataSize];
                        System.arraycopy(RxBuffer, 5, PacketData, 0, PacketData.length);
                        ProcessPacketData(PacketData);         
                        }
                }
                break;

            case HeaderTypeT.B:
                if (Receiving == false) {
                    if (DataByte == STX) {
                        // Begining of the packet
                        // Reset timeout timer
                        TimeOutTimer.restart();
                        RxBuffer[0] = STX;
                        ProcessedPacket = new ResponseArgs();        
                        Receiving = true;
                        ByteCounter = 1;          
                        DLEReceived = false;       
                    } else {
                        // Byte outside packet
                        // Stop timeout timer
                        TimeOutTimer.stop();
                    }
                } else {
                    if (ByteCounter == RxBuffer.length) {
                        // We have received too many bytes and exceeded RxBuffer size
                        // Stop timeout timer
                        TimeOutTimer.stop();
                        GenerateInvalidPacketEvent();
                        Receiving = false;
                        HeaderType = NewHeaderType;                        
                        break;
                    }
                    switch (DataByte) {
                        case STX: // error
                            // Stop timeout timer
                            TimeOutTimer.stop();
                            GenerateInvalidPacketEvent();
                            Receiving = false;
                            HeaderType = NewHeaderType;                            
                            break;

                        case ETX: // end of transmission
                            // Stop timeout timer
                            TimeOutTimer.stop();

                            if (DLEReceived == true || ByteCounter < 2) {
                                GenerateInvalidPacketEvent();
                                break;
                            }

                            ProcessedPacket.HeaderCRC = 0;
                            Receiving = false;
                            byte[] PacketData = new byte[ByteCounter - 1];
                            System.arraycopy(RxBuffer, 1, PacketData, 0, PacketData.length);
                            ProcessPacketData(PacketData);
                            break;

                        case DLE: // special sign
                            // Reset timeout timer
                            TimeOutTimer.restart();
                            DLEReceived = true;
                            break;

                        default:
                            // Reset timeout timer
                            TimeOutTimer.restart();
                            if (DLEReceived == true) {
                                DLEReceived = false;
                                RxBuffer[ByteCounter++] = (byte)(DataByte - DLE);
                            } else {
                                RxBuffer[ByteCounter++] = DataByte;
                            }
                            break;

                    }
                }
            break;
            }
        }
    }

    /// <summary>
    /// Function used to flush the Rx Buffer and return bytes which left.
    /// </summary>
    /// <returns> Method returns byte array or null when there are not bytes in the Rx Buffer. </returns>
    public byte[] FlushRxBuffer() {
        if (ByteCounter == 0) {
           return null;
        }

        byte[] Data = new byte[ByteCounter];
        System.arraycopy(RxBuffer, 0, Data, 0, Data.length);
        return Data;
    } 


    /// <summary>
    /// Method used to calculate and store CRC16 on a byte array.
    /// </summary>
    /// <param name="Data"> Input/output parameter representing the data bytes array where the CRC calculation will be done. CRC will be stored at the end of calculation byte stream. </param>
    /// <param name="Index"> Input parameter representing intex of the first byte in the calculations. </param>
    /// <param name="Count"> Input parameter representing number of bytes for the CRC to calculate. </param>
    private void Crc(byte[] Data, int Index, int Count) {
        char CRC, Temp;

        if (Index + Count + 2 > Data.length) {
            throw new UnsupportedOperationException("[Crc]: Not enough space to store crc.");
        }

        if (Count == 0) {
            throw new UnsupportedOperationException("[Crc]: Parameter Count has to be larger than 0.");
        }

        CRC = 0xFFFF;

        for (int i = 0; i < Count; ++i) {
            Temp = (char)(((CRC >> 8) ^(char) Data[Index + i] ) & (char)(0x00FF));
            CRC = (char) (CCITTCRCTable[Temp] ^ (char)((CRC << 8) & 0x0000FFFF)); //((UInt32)CRC << 8) & 0x0000FFFF));
        }

        Data[Index + Count] =     (byte)((CRC ) & 0x00FF);
        Data[Index + Count + 1] = (byte)((CRC >> 8) & 0x00FF);
    }

    /// <summary>
    /// Method used to calculate CRC16 on a byte array.
    /// </summary>
    /// <param name="Data"> Input parameter representing the data bytes array where the CRC calculation will be done. </param>
    /// <param name="Index">Input parameter representing intex of the first byte in the calculations. </param>
    /// <param name="Count">Input parameter representing number of bytes for the CRC to calculate. </param>
    /// <returns> Method returns calculated CRC16. </returns>
    private char GetCRC(byte[] Data, int Index, int Count) {
        char CRC, Temp;

        if (Index + Count > Data.length || Count == 0) {
            throw new UnsupportedOperationException("[GetCRC]: Invalid input parameters.");
        }

        CRC = 0xFFFF;

        for (int i = 0; i < Count; ++i) {
            Temp = (char)(((CRC >> 8) ^(char) Data[Index + i] ) & (char)(0x00FF));
            CRC = (char) (CCITTCRCTable[Temp] ^ (char)(((int)CRC << 8) & 0x0000FFFF));
        }

        return CRC;
    }

            /// <summary>
    /// Method used to check is the given character a special one (STX, ETX or DLE).
    /// </summary>
    /// <param name="c"> Input parameter - character to check for being special. </param>
    /// <returns> Method returns true if the given character is STX, ETX or DLE. False if not.</returns>
    private boolean IsSpecialChar(byte c) {
        return c == STX || c == ETX || c == DLE;
    }

    /// <summary>
    /// Method used to send packet with Type A header.
    /// </summary>
    /// <param name="Data"> Input parameter - array of bytes representing packet data. </param>
    private void SendPacketA(byte[] Data) {	        
        byte[] Packet = new byte[Data.length + 5];
        Packet[0] = 0x02;
        Packet[1] = (byte)((Data.length ) & 0x00FF);
        Packet[2] = (byte)((Data.length >> 8) & 0x00FF);
        Crc(Packet, 0, 3);
        System.arraycopy(Data, 0, Packet, 5, Data.length);
        Request(new RequestArgs(Packet));
    }

            /// <summary>
    /// Method used to send packet with Type B header.
    /// </summary>
    /// <param name="Data"> Input parameter - array of bytes representing packet data. </param>
    private void SendPacketB(byte[] Data) {
        int PacketSize = 2 + Data.length;

        for (int i = 0; i < Data.length; ++i) {
            if (IsSpecialChar(Data[i])) {
                ++PacketSize;
            }
        }

        byte[] Packet = new byte[PacketSize];
        Packet[0] = STX;
        Packet[PacketSize - 1] = ETX;

        int index = 1;

        for (int i = 0; i < Data.length; ++i) {
            if (IsSpecialChar(Data[i])) {
                Packet[index++] = DLE;
                Packet[index++] = (byte)(Data[i] + DLE);
            } else {
                Packet[index++] = Data[i];
            }
        }
        Request(new RequestArgs(Packet));
    }

            /// <summary>
    /// Method used to enrypt data.
    /// </summary>
    /// <param name="Data"> Input parameter - array of bytes representing packet data. </param>
    /// <returns> Function returns encrypted data as array of bytes. </returns>
    private byte[] Encrypt(byte[] Data) {
        try {
            IvParameterSpec iv = new IvParameterSpec(AesIv);
            SecretKeySpec skeySpec = new SecretKeySpec(AesKey, "AES");

            Cipher cipher = Cipher.getInstance("AES/CBC/NOPADDING");
            cipher.init(Cipher.ENCRYPT_MODE, skeySpec, iv);

            byte[] encrypted = cipher.doFinal(Data);

            return encrypted;
        } catch (Exception ex) {
            return null;
        }
    }

    private byte[] Decrypt(byte[] PacketData) {
        try {
            IvParameterSpec iv = new IvParameterSpec(AesIv);
            SecretKeySpec skeySpec = new SecretKeySpec(AesKey, "AES");

            Cipher cipher = Cipher.getInstance("AES/CBC/NOPADDING");
            cipher.init(Cipher.DECRYPT_MODE, skeySpec, iv);

            byte[] original = cipher.doFinal(PacketData);

            return original;
        } catch (Exception ex) {
            return null;
        }
    }  

     /// <summary>
    /// Method used to prepare the packet data for sending.
    /// </summary>
    /// <param name="Command"> Input parameter - byte representing the command. </param>
    /// <param name="Data"> Input parameter - array of bytes representing parameters. </param>
    /// <returns> Function returns array of bytes; packet data. </returns>
    private byte[] PreparePackedData(byte Command, byte[] Data) {
        char DataSize;
        if (Data == null || Data.length == 0) {
            DataSize = 1;
        } else {            
           DataSize = (char)(Data.length + 1);
        }


        switch (DataType) {
            case DataTypeT.Plain:
                byte[] PacketData = new byte[DataSize + 2]; // data + command + crc       
                PacketData[0] = Command;
                if (DataSize > 1) {
                   System.arraycopy(Data, 0, PacketData, 1, DataSize - 1);
                }
                Crc(PacketData, 0, DataSize);
                return PacketData;

            case DataTypeT.Encrypted:
                char FillSize;
                DataSize += 4;

                if (DataSize % 16 != 0) {
                    char NewSize;
                    NewSize = (char)(DataSize / 16);
                    NewSize += 1;
                    NewSize *= 16;
                    FillSize = (char)(NewSize - DataSize);
                } else {
                    FillSize = 0;
                }

                byte[] PlainData = new byte[DataSize + FillSize];

                PlainData[0] = (byte)((DataSize) & 0x00FF);
                PlainData[1] = (byte)((DataSize >> 8) & 0x00FF);
                PlainData[2] = Command;
                if (DataSize > 5) {
                    System.arraycopy(Data, 0, PlainData, 3, DataSize - 5);
                }
                Crc(PlainData, 0, DataSize - 2);

                if (FillSize > 0) {
                    Random r = new Random();
                    byte[] randomData = new byte[FillSize]; 
                    r.nextBytes(randomData);
                    System.arraycopy(randomData, 0, PlainData, DataSize, FillSize);
                }

                byte[] EncryptedData = Encrypt(PlainData);

                return EncryptedData;
        }

        return null;
    }
            /// <summary>
    /// Method used to send a command to the B1 module.
    /// </summary>
    /// <param name="Command"> Input parameter - command to be send to the B1 module. </param>
    /// <param name="Data"> Input parameter - array of bytes representing parameters. </param>
    private void SendCommand(byte Command, byte[] Data) {
    byte[] PacketData = PreparePackedData(Command, Data);

        switch (HeaderType) {
            case HeaderTypeT.A:
                SendPacketA(PacketData);
                break;

            case HeaderTypeT.B:
                SendPacketB(PacketData);
                break;
        }
    }

    public void SendDummyCommand() {
        SendCommand((byte)(UartCommandT.Dummy), null);
    }

    public void SendWriteToRfidMemoryCommand(int Address, byte[] Data) {	        
        if (Data.length == 0) {
            throw new UnsupportedOperationException("[SendWriteToRFIDMemoryCommand]: Data length is equal to 0.");
        }

        byte[] Parameters = new byte[Data.length + 4];
        Parameters[0] = (byte)((Address) & 0x00FF);
        Parameters[1] = (byte)((Address >> 8) & 0x00FF);
        Parameters[2] = (byte)((Data.length) & 0x00FF);
        Parameters[3] = (byte)((Data.length >> 8) & 0x00FF);

        System.arraycopy(Data, 0, Parameters, 4, Data.length);
        SendCommand((byte)(UartCommandT.WriteToRfidMemory), Parameters);
    }

    public void SendReadFromRfidMemoryCommand(int Address, int DataSize) {	        
        byte[] Data = new byte[4];
        Data[0] = (byte)((Address) & 0x00FF);
        Data[1] = (byte)((Address >> 8) & 0x00FF);
        Data[2] = (byte)((DataSize) & 0x00FF);
        Data[3] = (byte)((DataSize >> 8) & 0x00FF);	        
        SendCommand((UartCommandT.ReadFromRfidMemory), Data);
    }

    public void SendEnterSleepModeCommand() {	        
        SendCommand((UartCommandT.EnterSleepMode), null);
    }

    public void SendResetCommand() {
        SendCommand((UartCommandT.Reset), null);
    }

    public void SendSetBaudRateCommand(int BaudRate) {
        byte[] Data = new byte[4];
        Data[0] = (byte)(BaudRate & 0xFF);
        Data[1] = (byte)((BaudRate & 0xFF00) >> 8);
        Data[2] = (byte)((BaudRate & 0xFF0000) >> 16);
        Data[3] = (byte)((BaudRate & 0xFF000000) >> 24);

        SendCommand((UartCommandT.SetBaudRate), Data);
    }

    public void SendSetDataTypeCommand(byte newType) {
        byte[] Data = new byte[1];
        Data[0] = (byte)newType;
        SendCommand((byte)(UartCommandT.SetDataType), Data);
    }

    public void SendSetHeaderTypeCommand(byte newType) {
        byte[] Data = new byte[1];
        Data[0] = (byte)newType;
        SendCommand((byte)(UartCommandT.SetHeaderType), Data);
    }
    
    public void SetPacketHeaderType(byte newType)
    {
        NewHeaderType = newType;
        
        if (!Receiving)
            HeaderType = newType;
    }
    
    public void SetPacketEncoding(byte newDataType, byte[] newAESKey, byte[] newAESInitializationVector)
{
    if (newDataType == DataTypeT.Encrypted)
    {
        System.arraycopy(newAESKey,0, AesKey,0, 16);
        System.arraycopy(newAESInitializationVector,0, AesIv,0, 16);
    }

    DataType = newDataType;
}
    
    public void SendSetIoStateCommand(byte Gpio, byte GpioState) {
        byte[] Data = new byte[2];
        Data[0] = (byte)Gpio;
        Data[1] = (byte)GpioState;
        SendCommand((byte)(UartCommandT.SetIoState), Data);
    }

    public void SendReadIoStateCommand(byte Gpio) {
        byte[] Data = new byte[1];
        Data[0] = (byte)Gpio;
        SendCommand((byte)(UartCommandT.ReadIoState), Data);
    }

    public void SendSetIoInterruptCommand(byte Gpio, byte GpioInterrupt) {
        byte[] Data = new byte[2];
        Data[0] = (byte)Gpio;
        Data[1] = (byte)GpioInterrupt;
        SendCommand((byte)(UartCommandT.SetIoInterrupt), Data);
    }

    public void SendMeasureVoltageCommand(byte AdcSource, byte VoltageFormat) {
        byte[] Data = new byte[2];
        Data[0] = (byte)AdcSource;
        Data[1] = (byte)VoltageFormat;
        SendCommand((byte)(UartCommandT.MeasureVoltage), Data);
    }

    public void SendMeasureDieTemperatureVoltage(byte TemperatureFormat) {
        byte[] Data = new byte[1];
        Data[0] = (byte)TemperatureFormat;
        SendCommand((byte)(UartCommandT.MeasureDieTemperature), Data);
    }

    private void SendSetIdacCurrentCommand_nA(int Current) {
        byte[] Data = new byte[5];
        Data[0] = 0;

        System.arraycopy(ByteBuffer.allocate(4).order(ByteOrder.LITTLE_ENDIAN).putInt(Current).array(), 0, Data, 1, 4);
        SendCommand((byte)(UartCommandT.SetIdacCurrent), Data);
    }

    private void SendSetIdacCurrentCommand_nA(float Current) {
    byte[] Data = new byte[5];
    Data[0] = 1;
    System.arraycopy(ByteBuffer.allocate(4).order(ByteOrder.LITTLE_ENDIAN).putFloat(Current).array(), 0, Data, 1, 4);
    SendCommand((byte)(UartCommandT.SetIdacCurrent), Data);
}

    private void SendSetIdacCurrentCommand_uA(float Current) {
    byte[] Data = new byte[5];
    Data[0] = 2;
    System.arraycopy(ByteBuffer.allocate(4).order(ByteOrder.LITTLE_ENDIAN).putFloat(Current).array(), 0, Data, 1, 4);
    SendCommand((byte)(UartCommandT.SetIdacCurrent), Data);
}

    public void SendSetIdacCurrentCommand(byte CurrentFormat, Object Current) {
        switch (CurrentFormat) {
            case CurrentFormatT.Float_nA:
                SendSetIdacCurrentCommand_nA((float)Current);
                break;

            case CurrentFormatT.Float_uA:
                SendSetIdacCurrentCommand_uA((float)Current);
                break;

            case CurrentFormatT.Int32_nA:
                SendSetIdacCurrentCommand_nA((int)Current);
                break;

            default:
                break;
        }
    }

    public void SendEnableComparatorCommand(byte ReferenceVoltage, 
                                            byte OutputPin,
                                            byte EdgeSensitivity,                                            
                                            byte PowerSupplyDivider) {

        if (PowerSupplyDivider > 0x3F) {
            throw new UnsupportedOperationException("[SendEnableComparatorCommand]: PowerSupplyDivider maximum value exceeded.");
        }

        byte[] Data = new byte[4];
        Data[0] = (byte)ReferenceVoltage;
        Data[1] = (byte)OutputPin;
        Data[2] = (byte)EdgeSensitivity;
        Data[3] = PowerSupplyDivider;
        SendCommand((byte)(UartCommandT.EnableComparator), Data);
    }

    public void SendDisableComparatorCommand() {
        SendCommand((byte)(UartCommandT.DisableComparator), null);
    }

    private void SendEnablePwmCommand_Hz(int Frequency, byte Gpio, byte DutyCycle) {
        if (Frequency == 0) {
            throw new UnsupportedOperationException("[SendEnablePwmCommand_Hz]: Frequency parameter is equal to 0.");
        }

        byte[] Data = new byte[7];
        Data[0] = (byte)Gpio;
        Data[1] = DutyCycle;
        Data[2] = 0x00;
        System.arraycopy(ByteBuffer.allocate(4).order(ByteOrder.LITTLE_ENDIAN).putInt(Frequency).array(), 0, Data, 3, 4);
        SendCommand((byte)(UartCommandT.EnablePWM), Data);
    }

    private void SendEnablePwmCommand_Hz(float Frequency, byte Gpio, byte DutyCycle) {
        if (Frequency == 0.0f) {
            throw new UnsupportedOperationException("[SendEnablePwmCommand_Hz]: Frequency parameter is equal to 0.");
        }

        byte[] Data = new byte[7];
        Data[0] = (byte)Gpio;
        Data[1] = DutyCycle;
        Data[2] = 0x01;
        System.arraycopy(ByteBuffer.allocate(4).order(ByteOrder.LITTLE_ENDIAN).putFloat(Frequency).array(), 0, Data, 3, 4);
        SendCommand((byte)(UartCommandT.EnablePWM), Data);
    }

    private void SendEnablePwmCommand_us(int Period, byte Gpio, byte DutyCycle) {
        if (Period == 0) {
            throw new UnsupportedOperationException("[SendEnablePwmCommand_us]: Period parameter is equal to 0.");
        }

        byte[] Data = new byte[7];
        Data[0] = (byte)Gpio;
        Data[1] = DutyCycle;
        Data[2] = 0x02;
        System.arraycopy(ByteBuffer.allocate(4).order(ByteOrder.LITTLE_ENDIAN).putInt(Period).array(), 0, Data, 3, 4);
        SendCommand((byte)(UartCommandT.EnablePWM), Data);
    }

    private void SendEnablePwmCommand_s(float Period, byte Gpio, byte DutyCycle) {
        if (Period == 0.0f) {
            throw new UnsupportedOperationException("[SendEnablePwmCommand_s]: Frequency parameter is equal to 0.");
        }

        byte[] Data = new byte[7];
        Data[0] = (byte)Gpio;
        Data[1] = DutyCycle;
        Data[2] = 0x03;
        System.arraycopy(ByteBuffer.allocate(4).order(ByteOrder.LITTLE_ENDIAN).putFloat(Period).array(), 0, Data, 3, 4);
        SendCommand((byte)(UartCommandT.EnablePWM), Data);
    }

    public void SendEnablePwmCommand(byte PWMConfigFormat, Object Value, byte Gpio, byte DutyCycle) {
        switch (PWMConfigFormat) {
            case PWMConfigFormatT.Float_Hz:
                SendEnablePwmCommand_Hz((float)Value, Gpio, DutyCycle);
                break;

            case PWMConfigFormatT.Float_s:
                SendEnablePwmCommand_s((float)Value, Gpio, DutyCycle);
                break;

            case PWMConfigFormatT.Uint32_Hz:
                SendEnablePwmCommand_Hz((int)Value, Gpio, DutyCycle);
                break;

            case PWMConfigFormatT.Uint32_us:
                SendEnablePwmCommand_us((int)Value, Gpio, DutyCycle);
                break;

            default:
                break;
        }
    }

    public void SendSetAesInitVectorCommand(byte[] InitVector) {
        if (InitVector == null || InitVector.length != 16) {
            throw new UnsupportedOperationException("[SendSetAesInitVectorCommand]: Initialization Vector is not 16 bytes long.");
        }

        SendCommand((byte)(UartCommandT.SetAesInitVector), InitVector);
    }

    public void SendSetAesKeyCommand(byte[] AesKey) {
        if (AesKey == null || AesKey.length != 16) {
            throw new UnsupportedOperationException("[SendSetAesKeyCommand]: AES Key is not 16 bytes long.");
    }

    SendCommand((byte)(UartCommandT.SetAesKey), AesKey);
}

    public void SendReadAesInitVectorCommand() {
        SendCommand((byte)(UartCommandT.ReadAesInitVector), null);
    }

    public void SendReadAesKeyCommand() {
        SendCommand((byte)(UartCommandT.ReadAesKey), null);
    }

    public void SendWakeUpByte() {
        byte[] Data = new byte[1];
        Data[0] = 0x00;
        Request(new RequestArgs(Data));
    }

    public void SendStxByte() {
        byte[] Data = new byte[1];
        Data[0] = STX;
        Request(new RequestArgs(Data));
    }
}
