﻿/*
 * Created by Damian Gowor (Eccel Technology Ltd).
 * Release Date: 19/04/2017
 * Version : 1.2
 */
using System;
using System.IO;
using System.IO.Ports;
using System.Threading;
using System.Security.Cryptography;
using System.Collections;
using System.Collections.Generic;


namespace B1Client
{
    /// <summary>
    /// Enumeration representing UART commands.
    /// </summary>
    public enum UartCommandT {
        Dummy = 0x00,
        WriteToRfidMemory = 0x01,
        ReadFromRfidMemory = 0x02,
        EnterSleepMode = 0x03,
        Reset = 0x04,
        SetBaudRate = 0x05,
        SetDataType = 0x06,
        SetHeaderType = 0x07,
        SetIoState = 0x08,
        ReadIoState = 0x09,
        SetIoInterrupt = 0x0A,
        MeasureVoltage = 0x0B,
        MeasureDieTemperature = 0x0C,
        SetIdacCurrent = 0x0D,
        EnableComparator = 0x0E,
        DisableComparator = 0x0F,
        EnablePWM = 0x10,
        SetAesInitVector = 0x11,
        SetAesKey = 0x12,
        ReadAesInitVector = 0x13,
        ReadAesKey = 0x14
    }
    /// <summary>
    /// Enumeration respresenting the possible responses from B1 module as described in User Manual. Last two 0xFE and 0xFF are not directly send from B1 module.
    /// They are generated by this class when invalid packet is received or when timeout is generater per the specification.
    /// </summary>
    public enum ResponseT {
        ACK = 0,
        InvalidCommand = 0x01,
        InvalidParameter = 0x02,
        ProtocolError = 0x03,
        MemoryError = 0x04,
        SystemError = 0x05,
        ModuleTimeout = 0x06,
        Overflow = 0x07,
        AsyncPacket = 0x08,
        Busy = 0x09,
        SystemStart = 0x0A,
        
        InvalidPacket = 0xFE,
        ClientTimeout = 0xFF
    }
    
    /// <summary>
    /// Class used as argument when Response event is raised.
    /// </summary>
    public class ResponseArgs : EventArgs {
        public ResponseT Response;      
        public byte[] Parameters;
        public UInt16 HeaderCRC;
        public UInt16 DataCRC;
        
        public ResponseArgs(ResponseT Response, byte[] Parameters, UInt16 HeaderCRC, UInt16 DataCRC) : base () {
            this.Response   = Response;
            this.Parameters = Parameters;
            this.HeaderCRC  = HeaderCRC;
            this.DataCRC    = DataCRC;
        }
        
        public ResponseArgs() : base () {
            this.Response   = 0;
            this.Parameters = null;
            this.HeaderCRC  = 0;
            this.DataCRC    = 
                0;
        }
    }
    
    /// <summary>
    /// Class used as argument when Request event is raised.
    /// </summary>
    public class RequestArgs : EventArgs {
        public byte[] Data;
        
        public RequestArgs(byte[] Data) : base () {
            this.Data = Data;
        }
    }
    
    /// <summary>
    /// Enumeration representing possible DataType configurations.
    /// </summary>
    public enum DataTypeT {
        Plain = 0,
        Encrypted = 1
    }
    
    /// <summary>
    /// Enumeratuon representing possible HeaderType configurations.
    /// </summary>
    public enum HeaderTypeT {
        A = 0,
        B = 1
    }
    
    /// <summary>
    /// Enumeration of the B1 IOs.
    /// </summary>
    public enum GpioT {
        Gpio0 = 0,
        Gpio1 = 1,
        Gpio2 = 2,
        Gpio3 = 3
    }
    
    /// <summary>
    /// Enumeratuon representing possible GpioState configurations.
    /// </summary>
    public enum GpioStateT {
        DisabledHiZ = 0,
        Input = 1,
        OutputLow = 2,
        OutputHigh = 3
    }
    
    /// <summary>
    /// Enumeratuon representing possible GpioInterrupt configurations.
    /// </summary>
    public enum GpioInterruptT {
        FallingEdge = 0,
        RisingEdge = 1,
        AnyEdge = 2,
        Disabled = 3
    }
    
    /// <summary>
    /// Enumeratuon representing possible AdcSource configurations.
    /// </summary>
    public enum AdcSourceT {
        AdcPin = 0,
        PowerSupply = 1
    }
    
    /// <summary>
    /// Enumeratuon representing possible VoltageFormat configurations.
    /// </summary>
    public enum VoltageFormatT {
        Uint32_mv = 0,
        Float_mV = 1,
        Float_V = 2
    }
    
    /// <summary>
    /// Enumeratuon representing possible CurrentFormat configurations.
    /// </summary>
    public enum CurrentFormatT {
        Int32_nA = 0,
        Float_nA = 1,
        Float_uA = 2
    }
    
    /// <summary>
    /// Enumeratuon representing possible PWMConfigFormat configurations.
    /// </summary>
    public enum PWMConfigFormatT {
        Uint32_Hz = 0,
        Float_Hz = 1,
        Uint32_us = 2,
        Float_s = 3
    }
    
    /// <summary>
    /// Enumeratuon representing possible TemperatureFormat configurations.
    /// </summary>
    public enum TemperatureFormatT {
        Int32_mC = 0,
        Float_mC = 1,
        Float_C = 2,
        Int32_mF = 3,
        Float_mF = 4,
        Float_F = 5
    }
    
    /// <summary>
    /// Enumeratuon representing possible ComparatorReferenceVoltage configurations.
    /// </summary>
    public enum ComparatorReferenceVoltageT {
        Ref1V25 = 0,
        Ref2V5 = 1,
        RefDivVdd = 2
    }
    
    /// <summary>
    /// Enumeratuon representing possible ComparatorOutputPin configurations.
    /// </summary>
    public enum ComparatorOutputPinT {
        Disabled = 0,
        Enabled = 1,
        EnabledNegated = 2
    }
    
    /// <summary>
    /// Enumeratuon representing possible ComparatorAsyncPacketEdgeSensitivity configurations.
    /// </summary>
    public enum ComparatorAsyncPacketEdgeSensitivityT {
        Disabled = 0,
        FallingEdge = 1,
        RisingEdge = 2,
        AnyEdge = 3
    }
   
    /// <summary>
    /// Interface to the B1Client module.
    /// </summary>
    interface IB1Client {
        event EventHandler Response;
        event EventHandler Request;
        
        void SupplyRxData(byte[] Data);
        byte[] FlushRxBuffer();
        
        void SendDummyCommand();
        void SendWriteToRFIDMemoryCommand(UInt16 Address, byte[] Data);
        void SendReadFromRFIDMemoryCommand(UInt16 Address, UInt16 DataSize);
        void SendEnterSleepModeCommand();
        void SendResetCommand();
        void SendSetBaudRateCommand(UInt32 BaudRate);
        void SendSetDataTypeCommand(DataTypeT DataType);
        void SendSetHeaderTypeCommand(HeaderTypeT HeaderType);
        void SendSetIoStateCommand(GpioT Gpio, GpioStateT GpioState);
        void SendReadIoStateCommand(GpioT Gpio);
        void SendSetIoInterruptCommand(GpioT Gpio, GpioInterruptT GpioInterrupt);
        void SendMeasureVoltageCommand(AdcSourceT AdcSource, VoltageFormatT VoltageFormat);
        void SendMeasureDieTemperatureVoltage(TemperatureFormatT TemperatureFormat);        
        void SendSetIdacCurrentCommand(CurrentFormatT CurrentFormat, object Current);        
        void SendEnableComparatorCommand(ComparatorReferenceVoltageT ReferenceVoltage, 
                                                ComparatorOutputPinT OutputPin,
                                                ComparatorAsyncPacketEdgeSensitivityT EdgeSensitivity,                                            
                                                byte PowerSupplyDivider);
        void SendDisableComparatorCommand();        
        void SendEnablePwmCommand(PWMConfigFormatT PWMConfigFormat, object Value, GpioT Gpio, byte DutyCycle);        
        void SendSetAesInitVectorCommand(byte[] InitVector);
        void SendSetAesKeyCommand(byte[] AesKey);
        void SendReadAesInitVectorCommand();
        void SendReadAesKeyCommand();
        void SendWakeUpByte();
        void SendStxByte();
        

        DataTypeT DataType {
            get ;
            set ;
        }
        HeaderTypeT HeaderType {
            get ;
            set ;
        }
    }
	/// <summary>
	/// Class used to manage the B1 module.
	/// </summary>
	/// <para>
    /// Class is not thread safe.     
    /// </para>
	public class B1Module : IB1Client {
	    
	    /// <summary>
	    /// Table used when calculating CRC16.
	    /// </summary>
	    private readonly UInt16[] CCITTCRCTable = {
            0x0000, 0x1021, 0x2042, 0x3063, 0x4084, 0x50a5, 
            0x60c6, 0x70e7, 0x8108, 0x9129, 0xa14a, 0xb16b, 
            0xc18c, 0xd1ad, 0xe1ce, 0xf1ef, 0x1231, 0x0210, 
            0x3273, 0x2252, 0x52b5, 0x4294, 0x72f7, 0x62d6, 
            0x9339, 0x8318, 0xb37b, 0xa35a, 0xd3bd, 0xc39c, 
            0xf3ff, 0xe3de, 0x2462, 0x3443, 0x0420, 0x1401, 
            0x64e6, 0x74c7, 0x44a4, 0x5485, 0xa56a, 0xb54b, 
            0x8528, 0x9509, 0xe5ee, 0xf5cf, 0xc5ac, 0xd58d, 
            0x3653, 0x2672, 0x1611, 0x0630, 0x76d7, 0x66f6, 
            0x5695, 0x46b4, 0xb75b, 0xa77a, 0x9719, 0x8738, 
            0xf7df, 0xe7fe, 0xd79d, 0xc7bc, 0x48c4, 0x58e5, 
            0x6886, 0x78a7, 0x0840, 0x1861, 0x2802, 0x3823, 
            0xc9cc, 0xd9ed, 0xe98e, 0xf9af, 0x8948, 0x9969, 
            0xa90a, 0xb92b, 0x5af5, 0x4ad4, 0x7ab7, 0x6a96, 
            0x1a71, 0x0a50, 0x3a33, 0x2a12, 0xdbfd, 0xcbdc, 
            0xfbbf, 0xeb9e, 0x9b79, 0x8b58, 0xbb3b, 0xab1a, 
            0x6ca6, 0x7c87, 0x4ce4, 0x5cc5, 0x2c22, 0x3c03, 
            0x0c60, 0x1c41, 0xedae, 0xfd8f, 0xcdec, 0xddcd, 
            0xad2a, 0xbd0b, 0x8d68, 0x9d49, 0x7e97, 0x6eb6, 
            0x5ed5, 0x4ef4, 0x3e13, 0x2e32, 0x1e51, 0x0e70, 
            0xff9f, 0xefbe, 0xdfdd, 0xcffc, 0xbf1b, 0xaf3a, 
            0x9f59, 0x8f78, 0x9188, 0x81a9, 0xb1ca, 0xa1eb, 
            0xd10c, 0xc12d, 0xf14e, 0xe16f, 0x1080, 0x00a1, 
            0x30c2, 0x20e3, 0x5004, 0x4025, 0x7046, 0x6067, 
            0x83b9, 0x9398, 0xa3fb, 0xb3da, 0xc33d, 0xd31c, 
            0xe37f, 0xf35e, 0x02b1, 0x1290, 0x22f3, 0x32d2, 
            0x4235, 0x5214, 0x6277, 0x7256, 0xb5ea, 0xa5cb, 
            0x95a8, 0x8589, 0xf56e, 0xe54f, 0xd52c, 0xc50d, 
            0x34e2, 0x24c3, 0x14a0, 0x0481, 0x7466, 0x6447, 
            0x5424, 0x4405, 0xa7db, 0xb7fa, 0x8799, 0x97b8, 
            0xe75f, 0xf77e, 0xc71d, 0xd73c, 0x26d3, 0x36f2, 
            0x0691, 0x16b0, 0x6657, 0x7676, 0x4615, 0x5634, 
            0xd94c, 0xc96d, 0xf90e, 0xe92f, 0x99c8, 0x89e9, 
            0xb98a, 0xa9ab, 0x5844, 0x4865, 0x7806, 0x6827, 
            0x18c0, 0x08e1, 0x3882, 0x28a3, 0xcb7d, 0xdb5c, 
            0xeb3f, 0xfb1e, 0x8bf9, 0x9bd8, 0xabbb, 0xbb9a, 
            0x4a75, 0x5a54, 0x6a37, 0x7a16, 0x0af1, 0x1ad0, 
            0x2ab3, 0x3a92, 0xfd2e, 0xed0f, 0xdd6c, 0xcd4d, 
            0xbdaa, 0xad8b, 0x9de8, 0x8dc9, 0x7c26, 0x6c07, 
            0x5c64, 0x4c45, 0x3ca2, 0x2c83, 0x1ce0, 0x0cc1, 
            0xef1f, 0xff3e, 0xcf5d, 0xdf7c, 0xaf9b, 0xbfba, 
            0x8fd9, 0x9ff8, 0x6e17, 0x7e36, 0x4e55, 0x5e74, 
            0x2e93, 0x3eb2, 0x0ed1, 0x1ef0 
            }; 
	    
	    private bool Receiving = false;
	    private const byte STX = 0x02;
	    private const byte ETX = 0x03;
	    private const byte DLE = 0x10;
	    private int ByteCounter;
	    private UInt16 RxDataSize;
	    private byte[] RxBuffer = new byte[65536];
	    private DataTypeT _DataType = DataTypeT.Plain;
	    private HeaderTypeT _HeaderType = HeaderTypeT.A;
	    private byte[] _AesKey = new byte[16];
	    private byte[] _AesIv = new byte[16];
	    private bool DLEReceived = false;
	    private System.Windows.Forms.Timer TimeOutTimer;
	    private const Int32 TimeOutValue_ms = 100;
	    private ResponseArgs ProcessedPacket;
	    
	    public event EventHandler Response = null;
	    public event EventHandler Request = null;
	    
	    /// <summary>
	    /// Property for the AES Key to set up. This Key is used by the class to encrypt and decrypt B1 protocol. It does not influence the B1 module configuration.
	    /// </summary>
	    public byte[] AESKey {
	        get {
	            return _AesKey;
	        }
	        set {
	            if ( ((byte[])value).Length != 16) {
	                throw new Exception("Key length has to be equal to 16.");
	            }
	            _AesKey = value;
	        }
	    }
	    
	    /// <summary>
        /// Property for the AES Initialization Vector to set up. This IV is used by the class to encrypt and decrypt B1 protocol. It does not influence the B1 module configuration.
        /// </summary>
	    public byte[] AESIV {
            get {
                return _AesIv;
            }
            set {
                if ( ((byte[])value).Length != 16) {
                    throw new Exception("IV length has to be equal to 16.");
                }
                _AesIv = value;
            }
        }
	    
        /// <summary>
        /// Property for the B1 protocol Data Type configuration. It does not influence the B1 module configuration just the class itself.
        /// </summary>
	    public DataTypeT DataType {
	        get {
	            return _DataType;
	        }
	        set {
	            _DataType = value;
	        }
	    }
	    
        /// <summary>
        /// Property for the B1 protocol Header Type configuration. It does not influence the B1 module configuration just the class itself.
        /// </summary>
	    public HeaderTypeT HeaderType {
            get {
                return _HeaderType;
            }
            set {
                _HeaderType = value;
            }
        }
	    
        /// <summary>
        /// Method used to raise Response event with Invalid Packet response type.
        /// </summary>
	    private void GenerateInvalidPacketEvent() {
	        Response(this, new ResponseArgs(ResponseT.InvalidPacket, null, 0, 0));
	    }
	    
        /// <summary>
        /// Method used to raise Response event with Client Timeout response type.
        /// </summary>
	    private void GenerateClientTimeoutEvent() {
            Response(this, new ResponseArgs(ResponseT.ClientTimeout, null, 0, 0));
        }
	    
	    /// <summary> Private method which receives packed data and process it. </summary>
	    /// <para> This method is called after receiving full packet with valid header. When the method is executed on a valid packet data it fills 
	    /// the ProcessedPacket.DataCRC private field. Method recognizes object configuration for the packet data. Method doesn't check for the minimum size of the 
	    /// packet data array as it is called from the SupplyRxData which checks for the size. </para>
	    /// <param name="PacketData"> Array of bytes representing packet data after receiving packet with valid header. The size of it shall be minimum 3 bytes. </param>
	    private void ProcessPacketData(byte[] PacketData) {
            switch (_DataType) {
                case DataTypeT.Plain:                    
                    if (CheckPlainDataCRC(PacketData, ref ProcessedPacket.DataCRC) == false) {
	                    // If the packet data CRC is not correct we are going to send invalid packet and finish execution of this command.
                        GenerateInvalidPacketEvent();
                        return;
                    }
                    
                    if (Enum.IsDefined(typeof(ResponseT), (int)PacketData[0]) == false) {
	                    // If the response is not recognized we are going to send invalid packet and finish execution of this command.
	                    GenerateInvalidPacketEvent();;
                        return;
                    }
	                
	                ProcessedPacket.Response = (ResponseT)PacketData[0];
	                
	                if (PacketData.Length > 3) {
	                    // When we have at least one byte of parameters then we have to create a new array for them.
	                    ProcessedPacket.Parameters = new byte[PacketData.Length - 3];
	                    Buffer.BlockCopy(PacketData, 1, ProcessedPacket.Parameters, 0, ProcessedPacket.Parameters.Length);
	                } else {
	                    // There are no parameters bytes in the packet data.
	                    ProcessedPacket.Parameters = null;	                    
	                }
                          
	                Response(this, ProcessedPacket);
                    break;
                    
                case DataTypeT.Encrypted:
                    if (PacketData.Length % 16 != 0) {
                        GenerateInvalidPacketEvent();
                        return;
                    }
                    
                    byte[] PlainData = Decrypt(PacketData);
                    
                    // indexing is count from 1
                    UInt16 LastValidIndex = (UInt16)((UInt16)PlainData[0] + (UInt16)((UInt16)PlainData[1] << 8));
                    
                    if (LastValidIndex < 5) {
                        GenerateInvalidPacketEvent();
                        return;
                    }
                    
                     if (CheckEncryptedDataCRC(PlainData, LastValidIndex, ref ProcessedPacket.DataCRC) == false) {
                        GenerateInvalidPacketEvent();
                        return;
                    }
                    
                    if (Enum.IsDefined(typeof(ResponseT), (int)PlainData[2]) == false) {
                        GenerateInvalidPacketEvent();
                        return;
                    }
                    
                    ProcessedPacket.Response = (ResponseT)PlainData[2];
                    
                    if (LastValidIndex > 5) {
                        // When we have at least one byte of parameters then we have to create a new array for them.
                        ProcessedPacket.Parameters = new byte[LastValidIndex - 5];
                        Buffer.BlockCopy(PlainData, 3, ProcessedPacket.Parameters, 0, ProcessedPacket.Parameters.Length);
                    } else {
                        // There are no parameters bytes in the packet data.
                        ProcessedPacket.Parameters = null;                        
                    }
                    
                    Response(this, ProcessedPacket);
                    break;
                    
                default:
                    return;
            }
            
        }
	    
	    /// <summary> Method used by the client to supply to the object received data from the B1 / USB-B1 module. </summary>
	    /// <para> When a packed is received inside this method the header is processed if the object is set up to Header Type A. Then packet data is extracted and 
	    /// passed to private method ProcessPacketData. Private field ProcessedPacket.HeaderCRC is filled. If the object is configured as Header Type B then 
	    /// ProcessedPacket.HeaderCRC is set to 0 and packet data is extracted and passed to the private method ProcessPacketData. 
	    /// Method checks for the minimum size of the packet data and for the correct header. </para>
	    /// <param name="Data"> Array of bytes received from the B1 / USB-B1 module. </param>
	    /// <returns>  NotInitialised - the object is not yet initialized (it doesn't have response handler set).
	    ///            InvalidParameter - array passed to the method is null
	    ///            OK - data was processed.
	    /// </returns>
	    public void SupplyRxData(byte[] Data) {	        
	        if (Response == null) {
	            throw new Exception("Response handler not initialized");
	        }
	        
	        if (Data == null) {
	            throw new Exception("Data is null");
	        }
	        
	        foreach (byte DataByte in Data) {
	            switch (HeaderType) {
                    case HeaderTypeT.A:
                        if (Receiving == false) {
                            if (DataByte == STX) {
	                            // Begining of the packet
	                            // Reset timeout timer
                                TimeOutTimer.Stop();
                                TimeOutTimer.Start();        
	                            ProcessedPacket = new ResponseArgs();
                                RxBuffer[0]     = STX; // we are storing STX as the Header CRC is calculated with STX
                                Receiving       = true;
                                ByteCounter     = 1;
                                RxDataSize      = 0;
                            } else {
	                            // Byte outside packet
                                // Stop timeout timer
                                TimeOutTimer.Stop();
                            }
                        } else {
	                         // Reset timeout timer
                            TimeOutTimer.Stop();
                            TimeOutTimer.Start(); 
	                        RxBuffer[ByteCounter++] = DataByte;
	                        
                            if (ByteCounter == 5) {
                                // We have received a full header
                                // PacketSize = STX + 2 bytes DataSize + 2 bytes header CRC + DataSize = ByteCounter                                
                                if (DecodeHeaderTypeA(RxBuffer, ref ProcessedPacket.HeaderCRC, ref RxDataSize) == false) {
                                    // Stop timeout timer
                                    TimeOutTimer.Stop();
                                    GenerateInvalidPacketEvent();
                                    Receiving = false;
                                    break;
                                }
                                
                                 if (RxDataSize < 3) {
                                    // Packet data has to be at least 3 bytes long - Command + 2 bytes CRC
                                    // Stop timeout timer
                                    TimeOutTimer.Stop();
                                    GenerateInvalidPacketEvent();
                                    Receiving = false;                                    
                                    break;
                                }                                                                                               
                            } else if (ByteCounter == RxBuffer.Length) {
                                // We have received too many bytes and exceeded RxBuffer size
                                // Stop timeout timer
                                TimeOutTimer.Stop();
                                GenerateInvalidPacketEvent();
                                Receiving = false;
                                break;
	                        } else if (ByteCounter == (5 + RxDataSize)) {
	                            // Stop timeout timer
                                TimeOutTimer.Stop();
                                Receiving = false;
                                
	                            byte[] PacketData = new byte[RxDataSize];
                                Buffer.BlockCopy(RxBuffer, 5, PacketData, 0, PacketData.Length);
                                ProcessPacketData(PacketData);         
	                        }
                        }
                        break;
                        
                    case HeaderTypeT.B:
                        if (Receiving == false) {
                            if (DataByte == STX) {
                                // Begining of the packet
                                // Reset timeout timer
                                TimeOutTimer.Stop();
                                TimeOutTimer.Start();
                                RxBuffer[0] = STX;
                                ProcessedPacket = new ResponseArgs();        
                                Receiving = true;
                                ByteCounter = 1;          
                                DLEReceived = false;       
                            } else {
                                // Byte outside packet
                                // Stop timeout timer
                                TimeOutTimer.Stop();
                            }
                        } else {
                            if (ByteCounter == RxBuffer.Length) {
                                // We have received too many bytes and exceeded RxBuffer size
                                // Stop timeout timer
                                TimeOutTimer.Stop();
                                GenerateInvalidPacketEvent();
                                Receiving = false;
                                break;
                            }
                            switch (DataByte) {
                                case STX: // error
                                    // Stop timeout timer
                                    TimeOutTimer.Stop();
                                    GenerateInvalidPacketEvent();
                                    Receiving = false;
                                    break;
                                    
                                case ETX: // end of transmission
                                    // Stop timeout timer
                                    TimeOutTimer.Stop();
                                    
                                    if (DLEReceived == true || ByteCounter < 2) {
                                        GenerateInvalidPacketEvent();
                                        break;
                                    }
                                    
                                    ProcessedPacket.HeaderCRC = 0;
                                    byte[] PacketData = new byte[ByteCounter - 1];
                                    Buffer.BlockCopy(RxBuffer, 1, PacketData, 0, PacketData.Length);
                                    ProcessPacketData(PacketData);
                                    break;
                                    
                                case DLE: // special sign
                                    // Reset timeout timer
                                    TimeOutTimer.Stop();
                                    TimeOutTimer.Start();
                                    DLEReceived = true;
                                    break;
                                    
                                default:
                                    // Reset timeout timer
                                    TimeOutTimer.Stop();
                                    TimeOutTimer.Start();
                                    if (DLEReceived == true) {
                                        DLEReceived = false;
                                        RxBuffer[ByteCounter++] = (byte)(DataByte - DLE);
                                    } else {
                                        RxBuffer[ByteCounter++] = DataByte;
                                    }
                                    break;
                                    
                            }
                        }
                        break;
                        
                    default:
                        break;
                }
	        }
	    }
	    
	    /// <summary>
	    /// Function used to flush the Rx Buffer and return bytes which left.
	    /// </summary>
	    /// <returns> Method returns byte array or null when there are not bytes in the Rx Buffer. </returns>
	    public byte[] FlushRxBuffer() {
	        if (ByteCounter == 0) {
	           return null;
	        }
	        
	        byte[] Data = new byte[ByteCounter];
	        Buffer.BlockCopy(RxBuffer, 0, Data, 0, Data.Length);
	        return Data;
	    }
	    
	    /// <summary>
	    /// Method used to decode the Type A header.
	    /// </summary>
	    /// <param name="Packet"> Input parameter which contains the header bytes starting from index 0. </param>
	    /// <param name="CRC"> Output parameter where the CRC of the received header will be stored (not the calculated). </param>
	    /// <param name="DataSize"> Output parameter where packed data size in bytes will be stored. </param>
	    /// <returns> Function returns true if the header was parsed with positive result or false if there was an error. </returns>
	    private bool DecodeHeaderTypeA(byte[] Packet, ref UInt16 CRC, ref UInt16 DataSize) {
	        CRC = (UInt16)((UInt16)Packet[3] + (UInt16)((UInt16)Packet[4] << 8));
            
            if (CRC != GetCRC(Packet, 0, 3)) {
                return false;
            }
	        
	        DataSize = (UInt16)((UInt16)Packet[1] + (UInt16)((UInt16)Packet[2] << 8));
            
            return true;
	    }
	    
	    /// <summary>
	    /// Function used to check the plain data CRC.
	    /// </summary>
	    /// <param name="PacketData"> Input parameter which contains packet data bytes. </param>
	    /// <param name="CRC"> Output parameter where the CRC of the data will be stored (not the calculated). </param>
	    /// <returns> Function returns true if the calculated CRC was the same as received or false if not. </returns>
	    private bool CheckPlainDataCRC(byte[] PacketData, ref UInt16 CRC) {
            CRC = (UInt16)((UInt16)PacketData[PacketData.Length - 2] + (UInt16)((UInt16)PacketData[PacketData.Length - 1] << 8));
            
            if (CRC != GetCRC(PacketData, 0, PacketData.Length - 2)) {
                return false;
            }
            
            return true;
	    }
	    
	    /// <summary>
	    /// Function used to check the encrypted data CRC.
	    /// </summary>
	    /// <param name="EncryptedData"> Input parameter which contains packet data bytes. </param>
	    /// <param name="LastValidIndex"> Input parameters which represents the last valid index in the array (counting from 1) or the number of valid bytes. </param>
	    /// <param name="CRC"> Output parameter where the CRC wof the data will be stored (not the calculated). </param>
	    /// <returns> Function returns true if the calculated CRC was the same as received or false if not. </returns>
	    private bool CheckEncryptedDataCRC(byte[] EncryptedData, UInt16 LastValidIndex, ref UInt16 CRC) {
            CRC = (UInt16)((UInt16)EncryptedData[LastValidIndex - 2] + (UInt16)((UInt16)EncryptedData[LastValidIndex - 1] << 8));
            
            if (CRC != GetCRC(EncryptedData, 0, LastValidIndex - 2)) {
                return false;
            }
            
            return true;
        }   
	    
	    /// <summary>
	    /// Method called every time the timer expires. This timer is used to generate Client Timeout.
	    /// </summary>
	    /// <param name="myObject"> Standard. </param>
	    /// <param name="myEventArgs"> Standard. </param>
	    private void TimerEventProcessor(Object myObject, EventArgs myEventArgs) {
	       Receiving = false;
	       GenerateClientTimeoutEvent();
	    }
	    
	    /// <summary>
	    /// Constructor sets up the AES Key and IV to zeroes and launches timer.
	    /// </summary>
	    public B1Module() {
            _AesKey = new byte[16];
            _AesIv = new byte[16];
            
            for (int i = 0; i < 16; ++i) {
                _AesKey[i] = 0;
                _AesIv[i] = 0;
            }
            
            TimeOutTimer            = new System.Windows.Forms.Timer();
            TimeOutTimer.Interval   = TimeOutValue_ms;
            TimeOutTimer.Enabled    = false;
            TimeOutTimer.Tick       += new EventHandler(TimerEventProcessor);
            TimeOutTimer.Tag        = "Timeout timer";
        }
	    
	    
	    /// <summary>
	    /// Method used to calculate and store CRC16 on a byte array.
	    /// </summary>
	    /// <param name="Data"> Input/output parameter representing the data bytes array where the CRC calculation will be done. CRC will be stored at the end of calculation byte stream. </param>
	    /// <param name="Index"> Input parameter representing intex of the first byte in the calculations. </param>
	    /// <param name="Count"> Input parameter representing number of bytes for the CRC to calculate. </param>
	    private void Crc(byte[] Data, int Index, int Count) {
	        UInt16 CRC, Temp;
	        
	        if (Index + Count + 2 > Data.Length) {
	            throw new Exception("[Crc]: Not enough space to store crc.");
	        }
	        
	        if (Count == 0) {
	            throw new Exception("[Crc]: Parameter Count has to be larger than 0.");
	        }
	        
	        CRC = 0xFFFF;
	        
	        for (int i = 0; i < Count; ++i) {
	            Temp = (UInt16)(((CRC >> 8) ^(UInt16) Data[Index + i] ) & (UInt16)(0x00FF));
	            CRC = (UInt16) (CCITTCRCTable[Temp] ^ (UInt16)(((UInt32)CRC << 8) & 0x0000FFFF));
	        }
	        
	        Buffer.BlockCopy(BitConverter.GetBytes(CRC), 0, Data, Index + Count, 2);
	    }
	    
	    /// <summary>
	    /// Method used to calculate CRC16 on a byte array.
	    /// </summary>
	    /// <param name="Data"> Input parameter representing the data bytes array where the CRC calculation will be done. </param>
	    /// <param name="Index">Input parameter representing intex of the first byte in the calculations. </param>
	    /// <param name="Count">Input parameter representing number of bytes for the CRC to calculate. </param>
	    /// <returns> Method returns calculated CRC16. </returns>
	    private UInt16 GetCRC(byte[] Data, int Index, int Count) {
	        UInt16 CRC, Temp;
            
            if (Index + Count > Data.Length || Count == 0) {
                throw new Exception("[GetCRC]: Invalid input parameters.");
            }
            
            CRC = 0xFFFF;
            
            for (int i = 0; i < Count; ++i) {
                Temp = (UInt16)(((CRC >> 8) ^(UInt16) Data[Index + i] ) & (UInt16)(0x00FF));
                CRC = (UInt16) (CCITTCRCTable[Temp] ^ (UInt16)(((UInt32)CRC << 8) & 0x0000FFFF));
            }
            
            return CRC;
	    }
	    
	    /// <summary>
	    /// Method used to check is the given character a special one (STX, ETX or DLE).
	    /// </summary>
	    /// <param name="c"> Input parameter - character to check for being special. </param>
	    /// <returns> Method returns true if the given character is STX, ETX or DLE. False if not.</returns>
	    private bool IsSpecialChar(byte c) {
	        return c == STX || c == ETX || c == DLE;
	    }
	    
	    /// <summary>
	    /// Method used to send packet with Type A header.
	    /// </summary>
	    /// <param name="Data"> Input parameter - array of bytes representing packet data. </param>
	    private void SendPacketA(byte[] Data) {	        
	        byte[] Packet = new byte[Data.Length + 5];
	        Packet[0] = 0x02;
	        Buffer.BlockCopy(BitConverter.GetBytes((UInt16)Data.Length), 0, Packet, 1, 2);
	        Crc(Packet, 0, 3);
	        Buffer.BlockCopy(Data, 0, Packet, 5, Data.Length);
	        Request(this, new RequestArgs(Packet));
	    }
	    
	    /// <summary>
	    /// Method used to send packet with Type B header.
	    /// </summary>
	    /// <param name="Data"> Input parameter - array of bytes representing packet data. </param>
	    private void SendPacketB(byte[] Data) {
	        int PacketSize = 2 + Data.Length;
	        
	        for (int i = 0; i < Data.Length; ++i) {
	            if (IsSpecialChar(Data[i])) {
	                ++PacketSize;
	            }
	        }
	        
	        byte[] Packet = new byte[PacketSize];
	        Packet[0] = STX;
	        Packet[PacketSize - 1] = ETX;
	        
	        int index = 1;
	        
	        for (int i = 0; i < Data.Length; ++i) {
	            if (IsSpecialChar(Data[i])) {
	                Packet[index++] = DLE;
	                Packet[index++] = (byte)(Data[i] + DLE);
	            } else {
	                Packet[index++] = Data[i];
	            }
	        }
	        Request(this, new RequestArgs(Packet));
	    }
	    
	    /// <summary>
	    /// Method used to enrypt data.
	    /// </summary>
	    /// <param name="Data"> Input parameter - array of bytes representing packet data. </param>
	    /// <returns> Function returns encrypted data as array of bytes. </returns>
	    private byte[] Encrypt(byte[] Data) {
	        RijndaelManaged AES = new RijndaelManaged();
	        
	        
	        AES.KeySize = 128;
	        AES.BlockSize = 128;
	        AES.Mode = CipherMode.CBC;
	        AES.Padding = PaddingMode.None;
	        AES.Key = _AesKey;
            AES.IV = _AesIv;
	        
	        
	        MemoryStream memoryStream = new MemoryStream();
	        CryptoStream cryptoStream = new CryptoStream(memoryStream, AES.CreateEncryptor(), CryptoStreamMode.Write);
	        
	        cryptoStream.Write(Data, 0, Data.Length);
            cryptoStream.FlushFinalBlock();
            
            Byte[] EncryptedData = memoryStream.ToArray();
            
            memoryStream.Close();
            cryptoStream.Close();
            AES.Clear();

            return EncryptedData;
	    }
	    
	    /// <summary>
	    /// Method used to decrypt data.
	    /// </summary>
	    /// <param name="Data"> Input parameter - array of bytes representing encrypted packet data. </param>
	    /// <returns>Function returns decrypted data as array of bytes. </returns>
	    private byte[] Decrypt(byte[] Data) {
	        RijndaelManaged AES = new RijndaelManaged();
            
            
            AES.KeySize = 128;
            AES.BlockSize = 128;
            AES.Mode = CipherMode.CBC;
            AES.Padding = PaddingMode.None;
            AES.Key = _AesKey;
            AES.IV = _AesIv;
            
            
            MemoryStream memoryStream = new MemoryStream();
            CryptoStream cryptoStream = new CryptoStream(memoryStream, AES.CreateDecryptor(), CryptoStreamMode.Write);
            
            cryptoStream.Write(Data, 0, Data.Length);
            cryptoStream.FlushFinalBlock();
            
            Byte[] DecryptedData = memoryStream.ToArray();
            
            memoryStream.Close();
            cryptoStream.Close();
            AES.Clear();

            return DecryptedData;
	    }
	    
	    /// <summary>
	    /// Method used to prepare the packet data for sending.
	    /// </summary>
	    /// <param name="Command"> Input parameter - byte representing the command. </param>
	    /// <param name="Data"> Input parameter - array of bytes representing parameters. </param>
	    /// <returns> Function returns array of bytes; packet data. </returns>
	    private byte[] PreparePackedData(byte Command, byte[] Data) {
	        UInt16 DataSize;
            if (Data == null) {
                DataSize = 1;
            } else {            
               DataSize = (UInt16)(Data.Length + 1);
            }
	        
	        
	        switch (_DataType) {
	            case DataTypeT.Plain:
	                byte[] PacketData = new byte[DataSize + 2]; // data + command + crc       
                    PacketData[0] = Command;
                    if (DataSize > 1) {
                       Buffer.BlockCopy(Data, 0, PacketData, 1, DataSize - 1);
                    }
                    Crc(PacketData, 0, DataSize);
                    return PacketData;
	                
	            case DataTypeT.Encrypted:
                    UInt16 FillSize;
                    DataSize += 4;
                    
                    if (DataSize % 16 != 0) {
                        UInt16 NewSize;
                        NewSize = (UInt16)(DataSize / 16);
                        NewSize += 1;
                        NewSize *= 16;
                        FillSize = (UInt16)(NewSize - DataSize);
                    } else {
                        FillSize = 0;
                    }
                    
                    byte[] PlainData = new byte[DataSize + FillSize];
                    
                    Buffer.BlockCopy(BitConverter.GetBytes((UInt16)DataSize), 0, PlainData, 0, 2);
                    PlainData[2] = Command;
                    if (DataSize > 5) {
                        Buffer.BlockCopy(Data, 0, PlainData, 3, DataSize - 5);
                    }
                    Crc(PlainData, 0, DataSize - 2);
                    
                    if (FillSize > 0) {
                        Random r = new Random();
                        for (int i = 0; i < FillSize; ++i) {
                            PlainData[DataSize + i] = (byte)r.Next(255);
                        }
                    }
                    
                    byte[] EncryptedData = Encrypt(PlainData);
                    
                    
                    return EncryptedData;
	        }
	        
	        return null;
	    }
	    
	    /// <summary>
	    /// Method used to send a command to the B1 module.
	    /// </summary>
	    /// <param name="Command"> Input parameter - command to be send to the B1 module. </param>
	    /// <param name="Data"> Input parameter - array of bytes representing parameters. </param>
	    private void SendCommand(byte Command, byte[] Data) {
            byte[] PacketData = PreparePackedData(Command, Data);
            
	        switch (_HeaderType) {
	            case HeaderTypeT.A:
	                SendPacketA(PacketData);
	                break;
	                
	            case HeaderTypeT.B:
	                SendPacketB(PacketData);
	                break;
	        }
	    }
	    

	    
	    public void SendDummyCommand() {
	        SendCommand((byte)(UartCommandT.Dummy), null);
        }
	    
	    public void SendWriteToRFIDMemoryCommand(UInt16 Address, byte[] Data) {	        
	        if (Data.Length == 0) {
	            throw new Exception("[SendWriteToRFIDMemoryCommand]: Data length is equal to 0.");
	        }
	        
	        byte[] Parameters = new byte[Data.Length + 4];
	        Buffer.BlockCopy(BitConverter.GetBytes(Address), 0, Parameters, 0, 2);
	        Buffer.BlockCopy(BitConverter.GetBytes(Data.Length), 0, Parameters, 2, 2);
	        Buffer.BlockCopy(Data, 0, Parameters, 4, Data.Length);
	        SendCommand((byte)(UartCommandT.WriteToRfidMemory), Parameters);
	    }
	    
	    public void SendReadFromRFIDMemoryCommand(UInt16 Address, UInt16 DataSize) {	        
	        byte[] Data = new byte[4];
	        
	        Buffer.BlockCopy(BitConverter.GetBytes(Address), 0, Data, 0, 2);
	        Buffer.BlockCopy(BitConverter.GetBytes(DataSize), 0, Data, 2, 2);	        
	        SendCommand((byte)(UartCommandT.ReadFromRfidMemory), Data);
	    }
	    
	    public void SendEnterSleepModeCommand() {	        
	        SendCommand((byte)(UartCommandT.EnterSleepMode), null);
	    }
	    
	    public void SendResetCommand() {
	        SendCommand((byte)(UartCommandT.Reset), null);
	    }
	    
	    public void SendSetBaudRateCommand(UInt32 BaudRate) {
	        SendCommand((byte)(UartCommandT.SetBaudRate), BitConverter.GetBytes(BaudRate));
	    }
	    
	    public void SendSetDataTypeCommand(DataTypeT DataType) {
	        byte[] Data = new byte[1];
	        Data[0] = (byte)DataType;
	        SendCommand((byte)(UartCommandT.SetDataType), Data);
	    }
	    
	    public void SendSetHeaderTypeCommand(HeaderTypeT HeaderType) {
	        byte[] Data = new byte[1];
            Data[0] = (byte)HeaderType;
            SendCommand((byte)(UartCommandT.SetHeaderType), Data);
	    }
	    
	    public void SendSetIoStateCommand(GpioT Gpio, GpioStateT GpioState) {
	        byte[] Data = new byte[2];
	        Data[0] = (byte)Gpio;
	        Data[1] = (byte)GpioState;
	        SendCommand((byte)(UartCommandT.SetIoState), Data);
	    }
	    
	    public void SendReadIoStateCommand(GpioT Gpio) {
	        byte[] Data = new byte[1];
	        Data[0] = (byte)Gpio;
	        SendCommand((byte)(UartCommandT.ReadIoState), Data);
	    }
	    
	    public void SendSetIoInterruptCommand(GpioT Gpio, GpioInterruptT GpioInterrupt) {
	        byte[] Data = new byte[2];
	        Data[0] = (byte)Gpio;
	        Data[1] = (byte)GpioInterrupt;
	        SendCommand((byte)(UartCommandT.SetIoInterrupt), Data);
	    }
	    
	    public void SendMeasureVoltageCommand(AdcSourceT AdcSource, VoltageFormatT VoltageFormat) {
	        byte[] Data = new byte[2];
	        Data[0] = (byte)AdcSource;
	        Data[1] = (byte)VoltageFormat;
	        SendCommand((byte)(UartCommandT.MeasureVoltage), Data);
	    }
	    
	    public void SendMeasureDieTemperatureVoltage(TemperatureFormatT TemperatureFormat) {
	        byte[] Data = new byte[1];
	        Data[0] = (byte)TemperatureFormat;
	        SendCommand((byte)(UartCommandT.MeasureDieTemperature), Data);
	    }
	    
	    private void SendSetIdacCurrentCommand_nA(Int32 Current) {
	        byte[] Data = new byte[5];
	        Data[0] = 0;
	        Buffer.BlockCopy(BitConverter.GetBytes(Current), 0, Data, 1, 4);
	        SendCommand((byte)(UartCommandT.SetIdacCurrent), Data);
	    }
	    
	    private void SendSetIdacCurrentCommand_nA(float Current) {
            byte[] Data = new byte[5];
            Data[0] = 1;
            Buffer.BlockCopy(BitConverter.GetBytes(Current), 0, Data, 1, 4);
            SendCommand((byte)(UartCommandT.SetIdacCurrent), Data);
        }
	    
	    private void SendSetIdacCurrentCommand_uA(float Current) {
            byte[] Data = new byte[5];
            Data[0] = 2;
            Buffer.BlockCopy(BitConverter.GetBytes(Current), 0, Data, 1, 4);
            SendCommand((byte)(UartCommandT.SetIdacCurrent), Data);
        }
	    
	    public void SendSetIdacCurrentCommand(CurrentFormatT CurrentFormat, object Current) {
	        switch (CurrentFormat) {
	            case CurrentFormatT.Float_nA:
	                SendSetIdacCurrentCommand_nA((float)Current);
	                break;
	                
	            case CurrentFormatT.Float_uA:
	                SendSetIdacCurrentCommand_uA((float)Current);
	                break;
	                
	            case CurrentFormatT.Int32_nA:
	                SendSetIdacCurrentCommand_nA((Int32)Current);
	                break;
	                
	            default:
	                break;
	        }
	    }
	    
	    public void SendEnableComparatorCommand(ComparatorReferenceVoltageT ReferenceVoltage, 
	                                            ComparatorOutputPinT OutputPin,
                                                ComparatorAsyncPacketEdgeSensitivityT EdgeSensitivity,                                            
	                                            byte PowerSupplyDivider) {
	        
	        if (PowerSupplyDivider > 0x3F) {
	            throw new Exception("[SendEnableComparatorCommand]: PowerSupplyDivider maximum value exceeded.");
	        }
	        
	        byte[] Data = new byte[4];
	        Data[0] = (byte)ReferenceVoltage;
	        Data[1] = (byte)OutputPin;
	        Data[2] = (byte)EdgeSensitivity;
	        Data[3] = PowerSupplyDivider;
	        SendCommand((byte)(UartCommandT.EnableComparator), Data);
	    }
	    
	    public void SendDisableComparatorCommand() {
	        SendCommand((byte)(UartCommandT.DisableComparator), null);
	    }
	    
	    private void SendEnablePwmCommand_Hz(UInt32 Frequency, GpioT Gpio, byte DutyCycle) {
	        if (Frequency == 0) {
	            throw new Exception("[SendEnablePwmCommand_Hz]: Frequency parameter is equal to 0.");
	        }
	        
	        byte[] Data = new byte[7];
	        Data[0] = (byte)Gpio;
	        Data[1] = DutyCycle;
	        Data[2] = 0x00;
	        Buffer.BlockCopy(BitConverter.GetBytes(Frequency), 0, Data, 3, 4);
	        SendCommand((byte)(UartCommandT.EnablePWM), Data);
	    }
	    
	    private void SendEnablePwmCommand_Hz(float Frequency, GpioT Gpio, byte DutyCycle) {
            if (Frequency == 0.0f) {
                throw new Exception("[SendEnablePwmCommand_Hz]: Frequency parameter is equal to 0.");
            }
            
            byte[] Data = new byte[7];
            Data[0] = (byte)Gpio;
            Data[1] = DutyCycle;
            Data[2] = 0x01;
            Buffer.BlockCopy(BitConverter.GetBytes(Frequency), 0, Data, 3, 4);
            SendCommand((byte)(UartCommandT.EnablePWM), Data);
        }
	    
	    private void SendEnablePwmCommand_us(UInt32 Period, GpioT Gpio, byte DutyCycle) {
            if (Period == 0) {
                throw new Exception("[SendEnablePwmCommand_us]: Period parameter is equal to 0.");
            }
            
            byte[] Data = new byte[7];
            Data[0] = (byte)Gpio;
            Data[1] = DutyCycle;
            Data[2] = 0x02;
            Buffer.BlockCopy(BitConverter.GetBytes(Period), 0, Data, 3, 4);
            SendCommand((byte)(UartCommandT.EnablePWM), Data);
        }
	    
	    private void SendEnablePwmCommand_s(float Period, GpioT Gpio, byte DutyCycle) {
            if (Period == 0.0f) {
                throw new Exception("[SendEnablePwmCommand_s]: Frequency parameter is equal to 0.");
            }
            
            byte[] Data = new byte[7];
            Data[0] = (byte)Gpio;
            Data[1] = DutyCycle;
            Data[2] = 0x03;
            Buffer.BlockCopy(BitConverter.GetBytes(Period), 0, Data, 3, 4);
            SendCommand((byte)(UartCommandT.EnablePWM), Data);
        }
	    
	    public void SendEnablePwmCommand(PWMConfigFormatT PWMConfigFormat, object Value, GpioT Gpio, byte DutyCycle) {
	        switch (PWMConfigFormat) {
	            case PWMConfigFormatT.Float_Hz:
	                SendEnablePwmCommand_Hz((float)Value, Gpio, DutyCycle);
	                break;
	                
	            case PWMConfigFormatT.Float_s:
	                SendEnablePwmCommand_s((float)Value, Gpio, DutyCycle);
	                break;
	                
	            case PWMConfigFormatT.Uint32_Hz:
	                SendEnablePwmCommand_Hz((UInt32)Value, Gpio, DutyCycle);
	                break;
	                
	            case PWMConfigFormatT.Uint32_us:
	                SendEnablePwmCommand_us((UInt32)Value, Gpio, DutyCycle);
	                break;
	                
	            default:
	                break;
	        }
	    }
	    
	    public void SendSetAesInitVectorCommand(byte[] InitVector) {
	        if (InitVector == null || InitVector.Length != 16) {
	            throw new Exception("[SendSetAesInitVectorCommand]: Initialization Vector is not 16 bytes long.");
	        }
	        
	        SendCommand((byte)(UartCommandT.SetAesInitVector), InitVector);
	    }
	    
	    public void SendSetAesKeyCommand(byte[] AesKey) {
            if (AesKey == null || AesKey.Length != 16) {
                throw new Exception("[SendSetAesKeyCommand]: AES Key is not 16 bytes long.");
            }
            
            SendCommand((byte)(UartCommandT.SetAesKey), AesKey);
        }
	    
	    public void SendReadAesInitVectorCommand() {
            SendCommand((byte)(UartCommandT.ReadAesInitVector), null);
	    }
	    
	    public void SendReadAesKeyCommand() {
            SendCommand((byte)(UartCommandT.ReadAesKey), null);
        }
	    
	    public void SendWakeUpByte() {
	        byte[] Data = new byte[1];
	        Data[0] = 0x00;
	        Request(this, new RequestArgs(Data));
	    }
	    
	    public void SendStxByte() {
            byte[] Data = new byte[1];
            Data[0] = STX;
            Request(this, new RequestArgs(Data));
	    }
	}
}
