#include <Arduino.h>
#include <CCITTCRC.h>
#include <RFIDB1ClientInterface.h>
#include <RFIDB1ClientProtocol.h>

#ifdef ESP8266
//pins definition for ESP8266 board like NodeMCU
#include <SoftwareSerial.h>
#define debugSerial Serial
SoftwareSerial b1Serial(13,15);
#define STATUS_LED 16

#elif ESP32
//pins definition for ESP32 board like DOIT DevKit v1
#include <HardwareSerial.h>
HardwareSerial Serial2(2);
#define debugSerial Serial
#define b1Serial Serial2
#define STATUS_LED LED_BUILTIN

#else
//for other platforms
#define debugSerial Serial
#define b1Serial Serial2
#define STATUS_LED LED_BUILTIN
#endif

RFIDB1_Interface *B1Interface;
bool terminated = false;
uint32_t last_command = 0xff;
uint32_t last_command_param = 0xff;
uint8_t tagType;

#define B1SerialBaudrate_Default 9600
#define MAX_FRAME_SIZE 64

//---------------------------------------------------------------------------------------------------------------------------------------------------------------
uint8_t rxBuff[MAX_FRAME_SIZE * 2];
uint8_t txBuff[MAX_FRAME_SIZE * 2];

/**
    @brief Function used to send prepared data to UART hardware
    @param[in] data - data
    @param[in] size - data size
    @details Function is called after every UART command.
*/
void handleRequest(uint8_t *data, uint16_t size)
{
  if (b1Serial.write(data, size) <  size)
    debugSerial.println("writePacket error writing!");
}

/**
    @brief Function used to pars incoming RFB1 data
    @param[in] data - data
    @param[in] size - data size
    @details In this function user can parse incoming data. Data does not contain any headers and is always unencrypted.
*/
void handleResponse(uint8_t *data, uint16_t size)
{
    uint32_t i;

    if (data[0] == B1ResponseACK)
    {
      debugSerial.print("<= ACK");
      if (size > 1)
      {
          debugSerial.print(":");
          for (i = 1; i < size; i++)
          {
            char msg[8];
            sprintf(msg, " 0x%02X", data[i]);
            debugSerial.print(msg);
          }

      }
      debugSerial.print("\n");
    }

    switch (data[0])
    {
    case B1ResponseACK:
    {
        switch(last_command)
        {
        case B1RequestCommandReadFromRFIDMemory:
            switch(last_command_param)
            {
                case B1RegisterTagUID:
                {
                    uint8_t command;
                    switch (data[11])
                    {
                    case TagType_NoTag:             debugSerial.print("<= TagType: NoTag");            break;
                    case TagType_NotComplete:       debugSerial.print("<= TagType: NotComplete");      break;
                    case TagType_Ultralight:        debugSerial.print("<= TagType: Ultralight");       break;
                    case TagType_UltralightEV1_80B: debugSerial.print("<= TagType: UltralightEV1_80B");break;
                    case TagType_UltralightEV1_164B:debugSerial.print("<= TagType: UltralightEV1_164B");break;
                    case TagType_ClassicMini:       debugSerial.print("<= TagType: ClassicMini");      break;
                    case TagType_Classic1K:         debugSerial.print("<= TagType: Classic1K");        break;
                    case TagType_Classic4K:         debugSerial.print("<= TagType: Classic4K");        break;
                    case TagType_NTAG203F:          debugSerial.print("<= TagType: NTAG203F");         break;
                    case TagType_NTAG210:           debugSerial.print("<= TagType: NTAG210");          break;
                    case TagType_NTAG212:           debugSerial.print("<= TagType: NTAG212");          break;
                    case TagType_NTAG213F:          debugSerial.print("<= TagType: NTAG213F");         break;
                    case TagType_NTAG216F:          debugSerial.print("<= TagType: NTAG216F");         break;
                    case TagType_NTAG213:           debugSerial.print("<= TagType: NTAG213");          break;
                    case TagType_NTAG215:           debugSerial.print("<= TagType: NTAG215");          break;
                    case TagType_NTAG216:           debugSerial.print("<= TagType: NTAG216");          break;
                    default:
                        debugSerial.print("<= TagType: Unknown");
                        break;
                    }
                    tagType = data[11];

                    if (data[12] > 0 && tagType != TagType_NoTag)
                    {
                        char msg[8];
                        debugSerial.print("\n<= Tag UID size:");
                        sprintf(msg,"%d\n", data[12]);
                        debugSerial.print(msg);
                        debugSerial.print("<= UID: ");
                        for (i=0; i < data[12]; ++i)
                        {
                          sprintf(msg, "0x%02X ", data[1+i]);
                          debugSerial.print(msg);
                        }
                        digitalWrite(STATUS_LED, HIGH);
                    }

                    debugSerial.print("\n");
                    debugSerial.print("=> WriteToRFIDMemory(Halt)\n");
                    command = B1RFIDCommandHalt;
                    B1Interface->SendWriteToRFIDMemoryCommand(B1RegisterCommand, &command, 1);
                    last_command = B1RequestCommandWriteToRFIDMemory;
                    last_command_param = B1RFIDCommandHalt;
                }
                break;
                default:
                    debugSerial.print("Response on ReadFromRFIDMemory in unknown state\n");
                    break;
            }
            break;
        default:
            //debugSerial.print("ACK on command: 0x%02X\n", last_command);
            break;
        }
    }
    break;
    case B1ResponseAsyncPacket:
    {
        switch(last_command)
        {
        case B1RequestCommandWriteToRFIDMemory:
        {
            if (data[1] & B1AsyncPacketParamBit_RFIDCommand)
                switch(last_command_param)
                {
                case B1RFIDCommandGetUIDAndType:
                    debugSerial.print("=> ReadFromRFIDMemory(TagUID)\n");
                    B1Interface->SendReadFromRFIDMemoryCommand(B1RegisterTagUID, 12);
                    last_command = B1RequestCommandReadFromRFIDMemory;
                    last_command_param = B1RegisterTagUID;
                    break;
                case B1RFIDCommandHalt:
                    break;
                }
        }
        break;
        }
    }
    break;
    default:
        debugSerial.print("Unknown state\n");
        break;
    }
}

void setup() {

  pinMode(STATUS_LED, OUTPUT);
  RFIDB1_InterfaceConfigurationT config;

  debugSerial.begin(115200);

  debugSerial.print("Initializing interface...\n");
  B1Interface = new RFIDB1_Interface();

  memset(&config, 0, sizeof(config));

  config.handleRequest = handleRequest;
  config.handleResponse = handleResponse;
  config.InputBuffer = rxBuff;
  config.InputBufferSize = sizeof(rxBuff);
  config.OutputBuffer = txBuff;
  config.OutputBufferSize = sizeof(txBuff);

  if (B1Interface->Initialise(&config) != B1StatusT_OK)
  {
      debugSerial.print("Unable to initialise RFIDB1_interface\n");
      return;
  }

  B1Interface->SetPacketHeaderType(HeaderTypeA);
  B1Interface->SetPacketEncoding(DataTypePlain, NULL, NULL);
  b1Serial.begin(B1SerialBaudrate_Default);
}

void loop() {
  uint16_t bytesToRead;
  static uint32_t last_read = 0;
  uint8_t buff[16];

  bytesToRead = b1Serial.available();
  if (bytesToRead > sizeof(buff))
    bytesToRead = sizeof(buff);

  if (bytesToRead > 0)
  {
    bytesToRead = b1Serial.readBytes(buff, bytesToRead);
    B1Interface->ParseIncomingData(buff, bytesToRead);
    last_read = millis();
  }

  if (millis() - last_read > 500)
  {
    uint8_t command;
    last_read = millis();
    digitalWrite(STATUS_LED, LOW);
    debugSerial.print("=> WriteToRFIDMemory(GetUIDAndType)\n");
    command = B1RFIDCommandGetUIDAndType;
    B1Interface->SendWriteToRFIDMemoryCommand(B1RegisterCommand, &command, 1);
    last_command = B1RequestCommandWriteToRFIDMemory;
    last_command_param = B1RFIDCommandGetUIDAndType;
  }
}
