#include <Arduino.h>
#include <CCITTCRC.h>
#include <RFIDB1ClientInterface.h>
#include <RFIDB1ClientProtocol.h>
#include <PubSubClient.h>

#ifdef ESP8266
//pins definition for ESP8266 board like NodeMCU
#include <ESP8266WiFi.h>
#include <SoftwareSerial.h>
#define debugSerial Serial
SoftwareSerial b1Serial(13,15);
#define STATUS_LED 16

#elif ESP32
#include <WiFi.h>
//pins definition for ESP32 board like DOIT DevKit v1
#include <HardwareSerial.h>
HardwareSerial Serial2(2);
#define debugSerial Serial
#define b1Serial Serial2
#define STATUS_LED LED_BUILTIN

#else
error: this example works only with ESP modules
#endif

// Update these with values suitable for your network.
const char* ssid = "...";
const char* password = "...";
const char* mqtt_server = "...";
const char* mqtt_user = "...";
const char* mqtt_password = "...";
const char* mqtt_intopic = "rfid/in";
const char* mqtt_outtopic = "rfid/out";

WiFiClient espClient;
PubSubClient client(espClient);

RFIDB1_Interface *B1Interface;
bool terminated = false;
uint32_t last_command = 0xff;
uint32_t last_command_param = 0xff;

#define B1SerialBaudrate_Default 9600
#define MAX_FRAME_SIZE 64

//---------------------------------------------------------------------------------------------------------------------------------------------------------------
uint8_t rxBuff[MAX_FRAME_SIZE * 2];
uint8_t txBuff[MAX_FRAME_SIZE * 2];

/**
    @brief Function used to send prepared data to UART hardware
    @param[in] data - data
    @param[in] size - data size
    @details Function is called after every UART command.
*/
void handleRequest(uint8_t *data, uint16_t size)
{
  if (b1Serial.write(data, size) <  size)
    debugSerial.println("writePacket error writing!");
}

/**
    @brief Function used to pars incoming RFB1 data
    @param[in] data - data
    @param[in] size - data size
    @details In this function user can parse incoming data. Data does not contain any headers and is always unencrypted.
*/
void handleResponse(uint8_t *data, uint16_t size)
{
  uint32_t i;

  switch (data[0])
  {
    case B1ResponseACK:
    {
      if (last_command == B1RequestCommandReadFromRFIDMemory)
      {
        uint8_t command;

        if (data[12] > 0 && data[11] != TagType_NoTag) //size and tag type
        {
            char msg[32];

            digitalWrite(STATUS_LED, HIGH);

            debugSerial.print("<= Tag UID size:");
            sprintf(msg,"%d\n", data[12]);
            debugSerial.print(msg);
            debugSerial.print("<= UID: ");

            for (i=0; i < data[12]; ++i)
            {
              sprintf(&msg[i*2], "%02X", data[1+i]);
            }
            debugSerial.print(msg);
            Serial.print("\nSend tag to MQTT server: ");
            Serial.println(msg);
            client.publish(mqtt_outtopic, msg);
            Serial.println("");
        }
        else
          client.publish(mqtt_outtopic, "No TAG");

        command = B1RFIDCommandHalt;
        B1Interface->SendWriteToRFIDMemoryCommand(B1RegisterCommand, &command, 1);
        last_command = B1RequestCommandWriteToRFIDMemory;
        last_command_param = B1RFIDCommandHalt;
      }
    }
    break;
    case B1ResponseAsyncPacket:
      if (data[1] & B1AsyncPacketParamBit_RFIDCommand)
      {
        if (last_command == B1RequestCommandWriteToRFIDMemory && last_command_param == B1RFIDCommandGetUIDAndType)
        {
          B1Interface->SendReadFromRFIDMemoryCommand(B1RegisterTagUID, 12);
          last_command = B1RequestCommandReadFromRFIDMemory;
          last_command_param = B1RegisterTagUID;
        }
      }
    break;
    default:
        debugSerial.print("Unknown state\n");
        break;
    }
}

//ESP and MQTT functions

void setup_wifi() {

  delay(10);
  // We start by connecting to a WiFi network
  debugSerial.println();
  debugSerial.print("Connecting to ");
  debugSerial.println(ssid);

  WiFi.begin(ssid, password);

  while (WiFi.status() != WL_CONNECTED) {
    delay(500);
    Serial.print(".");
  }
  debugSerial.println("WiFi connected");
  debugSerial.println("IP address: ");
  debugSerial.println(WiFi.localIP());
}

void mqtt_reconnect() {
  while (!client.connected()) {
    debugSerial.print("Attempting MQTT connection...");
    if (client.connect("RFIDB1_Client", mqtt_user, mqtt_password)) {
      debugSerial.println("connected");
    } else {
      debugSerial.print("failed, rc=");
      debugSerial.print(client.state());
      debugSerial.println(" try again in 5 seconds");
      delay(5000);
    }
  }
  debugSerial.println("Waiting for RFID tag...");
}

void setup() {

  pinMode(STATUS_LED, OUTPUT);
  RFIDB1_InterfaceConfigurationT config;

  debugSerial.begin(115200);
  setup_wifi();

  client.setServer(mqtt_server, 1883);

  debugSerial.print("Initializing interface...\n");
  B1Interface = new RFIDB1_Interface();

  memset(&config, 0, sizeof(config));

  config.handleRequest = handleRequest;
  config.handleResponse = handleResponse;
  config.InputBuffer = rxBuff;
  config.InputBufferSize = sizeof(rxBuff);
  config.OutputBuffer = txBuff;
  config.OutputBufferSize = sizeof(txBuff);

  if (B1Interface->Initialise(&config) != B1StatusT_OK)
  {
      debugSerial.print("Unable to initialise RFIDB1_interface\n");
      return;
  }

  B1Interface->SetPacketHeaderType(HeaderTypeA);
  B1Interface->SetPacketEncoding(DataTypePlain, NULL, NULL);
  b1Serial.begin(B1SerialBaudrate_Default);
}

void loop() {
  uint16_t bytesToRead;
  static uint32_t last_read = 0;
  uint8_t buff[16];

  bytesToRead = b1Serial.available();
  if (bytesToRead > sizeof(buff))
    bytesToRead = sizeof(buff);

  if (bytesToRead > 0)
  {
    bytesToRead = b1Serial.readBytes(buff, bytesToRead);
    B1Interface->ParseIncomingData(buff, bytesToRead);
    last_read = millis();
  }

  if (millis() - last_read > 500)
  {
    uint8_t command;
    last_read = millis();
    digitalWrite(STATUS_LED, LOW);
    command = B1RFIDCommandGetUIDAndType;
    B1Interface->SendWriteToRFIDMemoryCommand(B1RegisterCommand, &command, 1);
    last_command = B1RequestCommandWriteToRFIDMemory;
    last_command_param = B1RFIDCommandGetUIDAndType;
  }

  if (!client.connected()) {
    mqtt_reconnect();
  }
  client.loop();
}
