/*
  RFIDB1ClientInterface.cpp - this file is part of the RFID-B1 
  library for RFIDB1 modules by Eccel Technology Ltd.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

   Authors:  Marcin Baliniak, Piotr Szmelcer, Damian Gowor

   Version: 1.0

  See file LICENSE.txt for further informations on licensing terms.
*/


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>

#include "RFIDB1ClientInterface.h"
#include "RFIDB1ClientProtocol.h"

RFIDB1_Interface::RFIDB1_Interface()
{

}

/**
    @brief Function used to disable module via GPIO functions
    @return B1StatusT_OK
    @return B1StatusT_Timeout - if module does not disable in default WAKEUP/DOWN time
    @return B1StatusT_NotInitialized - if hardware functions are not initialised
    @details SetPlatformInterface() should be called before user can use this function
*/
RFIDB1_StatusT RFIDB1_Interface::DisableModule()
{
    int k;

    if (!this->setResetPin || !this->setPowerPin || !this->delayMs || !this->getNSleep)
        return B1StatusT_NotInitialized;

    //set RST HIGH to prevent reset factory settings
    this->setResetPin(1);
    this->setPowerPin(0);

    for (k = 0; k < B1TimeoutMs_Startup; k++)
    {
        if (this->getNSleep() == 0)
            return B1StatusT_OK;

        this->delayMs(1);
    }

    return B1StatusT_Timeout;
}

/**
    @brief Function used to enable module via GPIO functions
    @return B1StatusT_OK
    @return B1StatusT_Timeout - if module does not enable in default WAKEUP/DOWN time
    @return B1StatusT_NotInitialized - if hardware functions are not initialised
    @details SetPlatformInterface() should be called before user can use this function
*/
RFIDB1_StatusT RFIDB1_Interface::EnableModule()
{
    int k;

    if (!this->setResetPin || !this->setPowerPin || !this->delayMs || !this->getNSleep)
        return B1StatusT_NotInitialized;

    this->setResetPin(1);
    this->setPowerPin(1);

    for (k = 0; k < B1TimeoutMs_Startup; k++)
    {
        if (this->getNSleep() == 1)
            return B1StatusT_OK;

        this->delayMs(1);
    }

    return B1StatusT_Timeout;
}

/**
    @brief Function used to reset module via GPIO functions
    @return B1StatusT_OK
    @return B1StatusT_NotInitialized - if hardware functions are not initialised
    @details SetPlatformInterface() should be called before user can use this function
*/
RFIDB1_StatusT RFIDB1_Interface::HardResetModule()
{

    if (!this->setResetPin || !this->setPowerPin || !this->delayMs || !this->getNSleep)
        return B1StatusT_NotInitialized;

    this->setResetPin(0);

    this->delayMs(10);

    this->setResetPin(1);

    this->delayMs(B1TimeoutMs_Startup);

    return B1StatusT_OK;
}

/**
    @brief Function used to reset device to factory default module via GPIO functions
    @return B1StatusT_OK
    @return B1StatusT_NotInitialized - if hardware functions are not initialised
    @details SetPlatformInterface() should be called before user can use this function
    should be used in case of communication problems
*/
RFIDB1_StatusT RFIDB1_Interface::ResetToDefaults()
{
    int nSleep;

    if (!this->setResetPin || !this->setPowerPin || !this->delayMs || !this->getNSleep)
        return B1StatusT_NotInitialized;

    nSleep = this->getNSleep();

    if (DisableModule() != B1StatusT_OK)
        return B1StatusT_SubModuleError;

    if (HardResetModule() != B1StatusT_OK)
        return B1StatusT_SubModuleError;

    //if module was enabled before
    if (nSleep)
    {
        if (EnableModule() != B1StatusT_OK)
            return B1StatusT_SubModuleError;
    }

    return B1StatusT_OK;
}

/**
    @brief Function used to store pointer for user data. eg to keep device descriptor for serial device
    @param[in] data - Pointer to user data
*/
void RFIDB1_Interface::SetUserData(void *data)
{
    this->UserData = data;
}

/**
    @brief Function used to reset device to factory default module via GPIO functions
    @return - Pointer to user data
*/
void* RFIDB1_Interface::GetUserData()
{
    return this->UserData;
}


/**
    @brief Function used to change packet header type on the application side
    @param[in] PacketHeaderType - header type B1SerialHeaderConstruction_TypeA or B1SerialHeaderConstruction_TypeB
    @return B1StatusT_OK
    @return B1StatusT_InvalidParameters
*/
RFIDB1_StatusT RFIDB1_Interface::SetPacketHeaderType(HeaderTypeT PacketHeaderType)
{
    if (this->ProtocolInterface->SetPacketHeaderType(PacketHeaderType) !=RFIDB1ProtocolStatusT_OK)
        return B1StatusT_InvalidParameters;
    else
        return B1StatusT_OK;
}

/**
    @brief Function used to change packet encoding on the application side
    @param[in] PacketEncoding - encoding type B1SerialDataConstruction_Plain or B1SerialDataConstruction_Encrypted
    @param[in] AESKey - pointer to AES key
    @param[in] AESInitializationVector - pointer to AES vector
    @return B1StatusT_OK
    @return B1StatusT_InvalidParameters
    @detail Should be used after switching module to new encoding type using SetDataType function
*/

RFIDB1_StatusT RFIDB1_Interface::SetPacketEncoding(DataTypeT PacketEncoding, uint8_t *AESKey, uint8_t *AESInitializationVector)
{
    if (this->ProtocolInterface->SetPacketEncoding(PacketEncoding, AESKey, AESInitializationVector) !=RFIDB1ProtocolStatusT_OK)
        return B1StatusT_InvalidParameters;
    else
        return B1StatusT_OK;
}

/**
    @brief Function used to parse UART incoming data
    @param[in] data - data pinter
    @param[in] size - size of incoming data
    @return B1StatusT_OK - if any correct packet received
    @return B1StatusT_NoRxPackets
    @details If function decode correct B1 packet call user function handleResponse()
*/
RFIDB1_StatusT RFIDB1_Interface::ParseIncomingData(uint8_t *data, uint16_t size)
{
    uint8_t *buff;
    uint16_t length;
    RFIDB1ProtocolStatusT status;
    RFIDB1_StatusT res = B1StatusT_NoRxPackets;

    do
    {
        status = this->ProtocolInterface->GetB1Packet(&length, data, &size);
        if (status == RFIDB1ProtocolStatusT_OK)
        {
            res = B1StatusT_OK;
            buff = this->ProtocolInterface->GetRxDataPointer();
            this->handleResponse(buff, length);
        }
    } while (size > 0);

    return res;
}

/**
    @brief Internal function user to prepare packet and send if writePacked is set
    @param[in] size - data size already copied in to Tx buff
    @return B1StatusT_OK
    @return B1StatusT_SubModuleError
*/
RFIDB1_StatusT RFIDB1_Interface::SendPacket(uint16_t size)
{
    RFIDB1ProtocolStatusT res;

    res = this->ProtocolInterface->PutB1Packet(size);

    if (this->ProtocolInterface->GetHeaderSize() + size + 1 > this->ProtocolInterface->GetTxBuffSizeMax())
        return B1StatusT_TxDataOverflow;

    if (res != RFIDB1ProtocolStatusT_OK)
        return B1StatusT_SubModuleError;

    if (this->handleRequest)
        this->handleRequest(this->ProtocolInterface->GetTxBuff(), this->ProtocolInterface->GetTxBuffSize());

    return B1StatusT_OK;
}

/**
    @brief Function used to send dummy command packet
    @return B1StatusT_OK
    @return B1StatusT_SubModuleError
    @details Incoming ACK will be received via handleResponse.
*/
RFIDB1_StatusT RFIDB1_Interface::SendDummyCommand()
{
    uint8_t *buff = this->ProtocolInterface->GetTxDataPointer();
    buff[0] =B1RequestCommandDummy;

    return SendPacket(1);
}

/**
    @brief Function used to send write to RFID memory command.
    @param[in] Address - RFID module memory address.
    @param[in] TxData - Pointer to Data.
    @param[in] TxDataSize - Size of Data
    @return B1StatusT_OK
    @return B1StatusT_InvalidParameters
    @return B1StatusT_SubModuleError
    @details Incoming ACK will be received via handleResponse.
*/
RFIDB1_StatusT RFIDB1_Interface::SendWriteToRFIDMemoryCommand(const uint16_t Address, const uint8_t *TxData, const uint16_t TxDataSize)
{
    uint8_t *buff = this->ProtocolInterface->GetTxDataPointer();

    if (TxData == NULL || TxDataSize == 0)
        return B1StatusT_InvalidParameters;

    if (this->ProtocolInterface->GetHeaderSize() + TxDataSize + 1 > this->ProtocolInterface->GetTxBuffSizeMax())
        return B1StatusT_TxDataOverflow;
    //Move data in buffer.
    memcpy(&buff[5], TxData, TxDataSize);

    //send command B1RequestCommandWriteToRFIDMemory
    buff[0] = B1RequestCommandWriteToRFIDMemory;
    buff[1] = (uint8_t)(Address & 0xFF);
    buff[2] = (uint8_t)((Address & 0xFF00) >> 8);
    buff[3] = (uint8_t)(TxDataSize & 0xFF);
    buff[4] = (uint8_t)((TxDataSize & 0xFF00) >> 8);

    return SendPacket(TxDataSize+5);
}

/**
    @brief Function used to send read from RFID memory command.
    @param[in] Address - RFID module memory address.
    @param[in] BytesToRead - Bytes to read.
    @return B1StatusT_OK
    @return B1StatusT_InvalidParameters
    @return B1StatusT_SubModuleError
    @details Incoming ACK will be received via handleResponse.
*/
RFIDB1_StatusT RFIDB1_Interface::SendReadFromRFIDMemoryCommand(const uint16_t Address, uint16_t BytesToRead)
{
    uint8_t *buff = this->ProtocolInterface->GetTxDataPointer();

    if (BytesToRead == 0)
        return B1StatusT_InvalidParameters;

    buff[0] = B1RequestCommandReadFromRFIDMemory;
    buff[1] = (uint8_t)(Address & 0xFF);
    buff[2] = (uint8_t)((Address & 0xFF00) >> 8);
    buff[3] = (uint8_t)(BytesToRead & 0xFF);
    buff[4] = (uint8_t)((BytesToRead & 0xFF00) >> 8);

    return SendPacket(5);
}

/**
    @brief Function used to send EnterSleepMode command.
    @return B1StatusT_OK
    @return B1StatusT_SubModuleError
    @details Incoming ACK will be received via handleResponse.
*/
RFIDB1_StatusT RFIDB1_Interface::SendEnterSleepModeCommand()
{
    uint8_t *buff = this->ProtocolInterface->GetTxDataPointer();

    buff[0] = B1RequestCommandEnterSleepMode;

    return SendPacket(1);
}

/**
    @brief Function used to send SoftwareReset command.
    @return B1StatusT_OK
    @return B1StatusT_SubModuleError
    @details Incoming ACK will be received via handleResponse.
*/
RFIDB1_StatusT RFIDB1_Interface::SendSoftwareResetCommand()
{
    uint8_t *buff = this->ProtocolInterface->GetTxDataPointer();

    buff[0] = B1RequestCommandReset;

    return SendPacket(1);
}

/**
    @brief Function used to send SetBaudRate command.
    @param[in] BaudRate - new baudrate
    @return B1StatusT_OK
    @return B1StatusT_SubModuleError
    @details Incoming ACK will be received via handleResponse.
    User should wait 50ms after receiving ACK.
*/
RFIDB1_StatusT RFIDB1_Interface::SendSetBaudRateCommand(uint32_t BaudRate)
{
    uint8_t *buff = this->ProtocolInterface->GetTxDataPointer();

    buff[0] = B1RequestCommandSetBaudRate;
    buff[1] = (uint8_t)(BaudRate & 0xFF);
    buff[2] = (uint8_t)((BaudRate & 0xFF00) >> 8);
    buff[3] = (uint8_t)((BaudRate & 0xFF0000) >> 16);
    buff[4] = (uint8_t)((BaudRate & 0xFF000000) >> 24);

    return SendPacket(5);
}

/**
    @brief Function used to send SetDataType command.
    @param[in] DataType - new data type B1SerialDataConstruction_Plain or B1SerialDataConstruction_Encrypted
    @return B1StatusT_OK
    @return B1StatusT_SubModuleError
    @details Incoming ACK will be received via handleResponse.
    User should wait 50ms after receiving ACK.
*/
RFIDB1_StatusT RFIDB1_Interface::SendSetDataTypeCommand(DataTypeT DataType)
{
    uint8_t *buff = this->ProtocolInterface->GetTxDataPointer();

    buff[0] = B1RequestCommandSetDataType;
    buff[1] = DataType;

    return SendPacket(2);
}

/**
    @brief Function used to send SetHeaderType command.
    @param[in] HeaderType - new header type B1SerialHeaderConstruction_TypeA or B1SerialHeaderConstruction_TypeB
    @return B1StatusT_OK
    @return B1StatusT_SubModuleError
    @details Incoming ACK will be received via handleResponse.
    User should wait 50ms after receiving ACK.
*/
RFIDB1_StatusT RFIDB1_Interface::SendSetHeaderTypeCommand(HeaderTypeT HeaderType)
{
    uint8_t *buff = this->ProtocolInterface->GetTxDataPointer();

    buff[0] = B1RequestCommandSetHeaderType;
    buff[1] = HeaderType;

    return SendPacket(2);
}

/**
    @brief Function used to send SetAESInitVector command.
    @param[in] HeaderType - new header type B1SerialHeaderConstruction_TypeA or B1SerialHeaderConstruction_TypeB
    @return B1StatusT_OK
    @return B1StatusT_SubModuleError
    @details Incoming ACK will be received via handleResponse.
    User should wait 50ms after receiving ACK.
*/
RFIDB1_StatusT RFIDB1_Interface::SendSetAESInitVectorCommand(uint8_t *AESInitializationVector)
{
    uint8_t *buff = this->ProtocolInterface->GetTxDataPointer();

    if (this->ProtocolInterface->GetHeaderSize() + 16 + 1 > this->ProtocolInterface->GetTxBuffSizeMax())
        return B1StatusT_TxDataOverflow;

    buff[0] = B1RequestCommandSetAESInitVector;
    memcpy(&buff[1], AESInitializationVector, 16);

    return SendPacket(17);
}

/**
    @brief Function used to send SetAESKey command.
    @param[in] HeaderType - new header type B1SerialHeaderConstruction_TypeA or B1SerialHeaderConstruction_TypeB
    @return B1StatusT_OK
    @return B1StatusT_SubModuleError
    @details Incoming ACK will be received via handleResponse.
    User should wait 50ms after receiving ACK.
*/
RFIDB1_StatusT RFIDB1_Interface::SendSetAESKeyCommand(uint8_t *AESKey)
{
    uint8_t *buff = this->ProtocolInterface->GetTxDataPointer();

    if (this->ProtocolInterface->GetHeaderSize() + 16 + 1 > this->ProtocolInterface->GetTxBuffSizeMax())
        return B1StatusT_TxDataOverflow;

    buff[0] = B1RequestCommandSetAESKey;
    memcpy(&buff[1], AESKey, 16);

    return SendPacket(17);
}

/**
    @brief Function used to send SetIOState command.
    @param[in] HeaderType - new header type B1SerialHeaderConstruction_TypeA or B1SerialHeaderConstruction_TypeB
    @return B1StatusT_OK
    @return B1StatusT_SubModuleError
    @details Incoming ACK will be received via handleResponse.
    User should wait 50ms after receiving ACK.
*/
RFIDB1_StatusT RFIDB1_Interface::SendSetIOStateCommand(const GpioT Gpio, const GpioStateT GpioState)
{
    uint8_t *buff = this->ProtocolInterface->GetTxDataPointer();

    buff[0] = B1RequestCommandSetIOState;
    buff[1] = Gpio;
    buff[2] = GpioState;

    return SendPacket(3);
}

/**
    @brief Function used to send ReadIOState command.
    @param[in] IONumber - GPIO number.
    @return B1StatusT_OK
    @return B1StatusT_SubModuleError
    @details Incoming ACK will be received via handleResponse.
*/
RFIDB1_StatusT RFIDB1_Interface::SendReadIOStateCommand(const GpioT Gpio)
{
    uint8_t *buff = this->ProtocolInterface->GetTxDataPointer();

    buff[0] = B1RequestCommandReadIOState;
    buff[1] = Gpio;

    return SendPacket(2);
}

/**
    @brief Function used to send SetIOInterrupt command.
    @param[in] IONumber - GPIO number.
    @return B1StatusT_OK
    @return B1StatusT_SubModuleError
    @details Incoming ACK will be received via handleResponse.
*/
RFIDB1_StatusT RFIDB1_Interface::SendSetIOInterruptCommand(const GpioT Gpio, const GpioInterruptT GpioInterrupt)
{
    uint8_t *buff = this->ProtocolInterface->GetTxDataPointer();

    buff[0] = B1RequestCommandSetIOInterrupt;
    buff[1] = Gpio;
    buff[2] = GpioInterrupt;

    return SendPacket(3);
}

/**
    @brief Function used to send MeasureVoltage command.
    @param[in] VoltageSource - Source of voltage measurment.
    @param[in] VoltageFormat - Format of returned value.
    @return B1StatusT_OK
    @return B1StatusT_SubModuleError
    @details Incoming ACK will be received via handleResponse.
*/
RFIDB1_StatusT RFIDB1_Interface::SendMeasureVoltageCommand(const AdcSourceT AdcSource, const VoltageFormatT VoltageFormat)
{
    uint8_t *buff = this->ProtocolInterface->GetTxDataPointer();

    buff[0] = B1RequestCommandMeasureVoltage;
    buff[1] = AdcSource;
    buff[2] = VoltageFormat;

    return SendPacket(3);
}

/**
    @brief Function used to send MeasureDieTemperature command.
    @param[in] TemperatureFormat - Temperature format configuration byte.
    @return B1StatusT_OK
    @return B1StatusT_SubModuleError
    @details Incoming ACK will be received via handleResponse.
*/
RFIDB1_StatusT RFIDB1_Interface::SendMeasureDieTemperatureCommand(const TemperatureFormatT TemperatureFormat)
{
    uint8_t *buff = this->ProtocolInterface->GetTxDataPointer();

    buff[0] = B1RequestCommandMeasureDieTemperature;
    buff[1] = TemperatureFormat;

    return SendPacket(2);
}

/**
    @brief Function used to send SetIDACCurrent command.
    @param[in] CurrentFormat - Current format.
    @param[in] CurrentValue - pointer to Value for set.
    @return B1StatusT_OK
    @return B1StatusT_SubModuleError
    @details Incoming ACK will be received via handleResponse.
*/
RFIDB1_StatusT RFIDB1_Interface::SendSetIDACCurrentCommand(const CurrentFormatT CurrentFormat, void *CurrentValue)
{
    uint8_t *buff = this->ProtocolInterface->GetTxDataPointer();
    uint32_t *CurrentValueP = (uint32_t *)CurrentValue;

    buff[0] = B1RequestCommandSetIDACCurrent;
    buff[1] = CurrentFormat;
    buff[2] = (uint8_t)(*CurrentValueP & 0xFF);
    buff[3] = (uint8_t)((*CurrentValueP & 0xFF00) >> 8);
    buff[4] = (uint8_t)((*CurrentValueP & 0xFF0000) >> 16);
    buff[5] = (uint8_t)((*CurrentValueP & 0xFF000000) >> 24);

    return SendPacket(6);
}

/**
    @brief Function used to send EnableComparator command.
    @param[in] ComparatorInputConfig - Comparator input configuration byte.
    @param[in] PowerSupplyDividerValue - Power supply divider value (0-63).
    @param[in] ComparatorOutputConfig - Comparator output configuration byte.
    @return B1StatusT_OK
    @return B1StatusT_InvalidParameters
    @return B1StatusT_SubModuleError
    @details Incoming ACK will be received via handleResponse.
*/
RFIDB1_StatusT RFIDB1_Interface::SendEnableComparatorCommand(const ComparatorReferenceVoltageT InputConfig, const ComparatorOutputPinT OutputPinConfig, const ComparatorAsyncPacketEdgeSensitivityT PacketConfig, const uint8_t PowerSupplyDividerValue)
{
    uint8_t *buff = this->ProtocolInterface->GetTxDataPointer();

    if (PowerSupplyDividerValue > 0x3F)
        return B1StatusT_InvalidParameters;

    buff[0] = B1RequestCommandEnableComparator;
    buff[1] = InputConfig;
    buff[2] = OutputPinConfig;
    buff[3] = PacketConfig;
    buff[4] = PowerSupplyDividerValue;

    return SendPacket(5);
}

/**
    @brief Function used to send DisableComparator command.
    @return B1StatusT_OK
    @return B1StatusT_InvalidParameters
    @return B1StatusT_SubModuleError
    @details Incoming ACK will be received via handleResponse.
*/
RFIDB1_StatusT RFIDB1_Interface::SendDisableComparatorCommand()
{
    uint8_t *buff = this->ProtocolInterface->GetTxDataPointer();

    buff[0] = B1RequestCommandDisableComparator;

    return SendPacket(1);
}

/**
    @brief Function used to send EnablePWM command.
    @param[in] IONumber - GPIO number.
    @param[in] DutyCycle - Duty cycle in %.
    @param[in] ValueFormat - PWM frequency or period format.
    @param[in] Value - frequency or period value.
    @return B1StatusT_OK
    @return B1StatusT_SubModuleError
    @details Incoming ACK will be received via handleResponse.
*/

RFIDB1_StatusT RFIDB1_Interface::SendEnablePWMCommand(PWMConfigFormatT PWMConfigFormat, void *Value, GpioT Gpio, uint8_t DutyCycle)
{
    uint8_t *buff = this->ProtocolInterface->GetTxDataPointer();
    uint32_t *ValueP = (uint32_t*)Value;

    buff[0] = B1RequestCommandEnablePWM;
    buff[1] = Gpio;
    buff[2] = DutyCycle;
    buff[3] = PWMConfigFormat;
    buff[4] = (uint8_t)(*ValueP & 0xFF);
    buff[5] = (uint8_t)((*ValueP & 0xFF00) >> 8);
    buff[6] = (uint8_t)((*ValueP & 0xFF0000) >> 16);
    buff[7] = (uint8_t)((*ValueP & 0xFF000000) >> 24);

   return SendPacket(8);
}

/**
    @brief Function used to send ReadAESInitVector command.
    @return B1StatusT_OK
    @return B1StatusT_SubModuleError
    @details Incoming ACK will be received via handleResponse.
*/
RFIDB1_StatusT RFIDB1_Interface::SendReadAESInitVectorCommand()
{
    uint8_t *buff = this->ProtocolInterface->GetTxDataPointer();

    buff[0] = B1RequestCommandReadAESInitVector;

    return SendPacket(1);
}

/**
    @brief Function used to send ReadAESKey command.
    @return B1StatusT_OK
    @return B1StatusT_SubModuleError
    @details Incoming ACK will be received via handleResponse.
*/
RFIDB1_StatusT RFIDB1_Interface::SendReadAESKeyCommand()
{
    uint8_t *buff = this->ProtocolInterface->GetTxDataPointer();

    buff[0] = B1RequestCommandReadAESKey;

    return SendPacket(1);
}


/**
    @brief Function used to send raw command.
    @param[in] Data - pointer to data buffer.
    @param[in] Size - data size
    @return B1StatusT_OK
    @return B1StatusT_SubModuleError
    @details Incoming ACK will be received via handleResponse.
*/
RFIDB1_StatusT RFIDB1_Interface::SendRawDataCommand(uint8_t *Data, uint16_t Size)
{
    if (Size > this->ProtocolInterface->GetTxBuffSizeMax())
        return B1StatusT_TxDataOverflow;

    memcpy(this->ProtocolInterface->GetTxBuff(), Data, Size);
    this->ProtocolInterface->SetTxBuffSize(Size);

    if (this->handleRequest)
        this->handleRequest(this->ProtocolInterface->GetTxBuff(), this->ProtocolInterface->GetTxBuffSize());

    return B1StatusT_OK;
}

/**
    @brief Function used to send wake up command
    @return B1StatusT_OK
    @return B1StatusT_SubModuleError
    @details Incoming ACK will be received via handleResponse.
*/
RFIDB1_StatusT RFIDB1_Interface::SendWakeUpByte()
{
    (this->ProtocolInterface->GetTxBuff())[0] = 0x00;
    this->ProtocolInterface->SetTxBuffSize(1);

    if (this->handleRequest)
        this->handleRequest(this->ProtocolInterface->GetTxBuff(), this->ProtocolInterface->GetTxBuffSize());

    return B1StatusT_OK;

}

/**
    @brief Function used to send wake up command
    @return B1StatusT_OK
    @return B1StatusT_SubModuleError
    @details Incoming ACK will be received via handleResponse.
*/
RFIDB1_StatusT RFIDB1_Interface::SendStxByte()
{
    (this->ProtocolInterface->GetTxBuff())[0] = B1Header_STX;
    this->ProtocolInterface->SetTxBuffSize(1);

    if (this->handleRequest)
        this->handleRequest(this->ProtocolInterface->GetTxBuff(), this->ProtocolInterface->GetTxBuffSize());

    return B1StatusT_OK;

}

/**
    @brief Function used to initialise object
    @param[in] config - Pointer to RFIDB1 configuration
    @return B1StatusT_OK
    @return B1StatusT_InvalidParameters
    @detail This function should be called just after GetRFIDB1Interface() with full configuration for RFIDB1 interface
*/
RFIDB1_StatusT RFIDB1_Interface::Initialise(RFIDB1_InterfaceConfigurationT *config)
{
    RFIDB1_IProtocolConfigurationT protocolConfig;

    if (!config->handleRequest || !config->handleResponse)
        return B1StatusT_InvalidParameters;

    this->handleRequest = config->handleRequest;
    this->handleResponse = config->handleResponse;

    this->getNSleep = NULL;
    this->setResetPin = NULL;
    this->setPowerPin = NULL;
    this->delayMs = NULL;

    //optional functions
    if (config->getNSleep)
        this->getNSleep = config->getNSleep;

    if (config->setResetPin)
        this->setResetPin = config->setResetPin;

    if (config->setPowerPin)
        this->setPowerPin = config->setPowerPin;

    if (config->delayMs)
        this->delayMs = config->delayMs;


    this->ProtocolInterface = new RFIDB1ProtocolInterface();

    protocolConfig.InputBuffer = config->InputBuffer;
    protocolConfig.InputBufferSize = config->InputBufferSize;
    protocolConfig.OutputBuffer = config->OutputBuffer;
    protocolConfig.OutputBufferSize = config->OutputBufferSize;
    if (config->AesDecryptBuffer)
      protocolConfig.AesDecryptBuffer = config->AesDecryptBuffer;
    if (config->AesEncryptBuffer)
      protocolConfig.AesEncryptBuffer = config->AesEncryptBuffer;

    if (this->ProtocolInterface->Initialise(&protocolConfig) != RFIDB1ProtocolStatusT_OK)
        return B1StatusT_InvalidParameters;

    return B1StatusT_OK;
}
