/*
  RFIDB1ClientProtocol.cpp - this file is part of the RFID-B1 
  library for RFIDB1 modules by Eccel Technology Ltd.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

   Authors:  Marcin Baliniak, Piotr Szmelcer, Damian Gowor

   Version: 1.0

  See file LICENSE.txt for further informations on licensing terms.
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>


#include "RFIDB1ClientProtocol.h"
#include "CCITTCRC.h"

RFIDB1ProtocolInterface::RFIDB1ProtocolInterface()
{
  this->AesDecryptBuffer = NULL;
  this->AesEncryptBuffer = NULL;
}

/**
	@brief Function initialise protocol
	@param[in] config - protocol configuration
    @return RFIDB1ProtocolStatusT_OK.
    @return RFIDB1ProtocolStatusT_InvalidParameters - wrong input parameters.
	*/
RFIDB1ProtocolStatusT RFIDB1ProtocolInterface::Initialise(RFIDB1_IProtocolConfigurationT *config)
{
    if (!config->InputBuffer || !config->OutputBuffer || config->InputBufferSize < 10 || config->OutputBufferSize < 10)
        return RFIDB1ProtocolStatusT_InvalidParameters;

    this->TxBuff = config->InputBuffer;
    this->TxBuffSizeMax = config->InputBufferSize;
    this->RxBuff = config->OutputBuffer;
    this->RxBuffSizeMax = config->OutputBufferSize;
    this->AesDecryptBuffer = config->AesDecryptBuffer;
    this->AesEncryptBuffer = config->AesEncryptBuffer;
    this->Receiving = false;
    this->ByteCounter = 0;
    this->HeaderCRC = 0;
    this->DLEReceived = false;
    this->DataType   = DataTypePlain;
    return RFIDB1ProtocolStatusT_OK;
}


/**
    @brief  Function used to count DEL-bytes in buff.
    @param[in]  buff - pointer to input RAW data which will be DLE-modified.
    @param[in] size - pointer to a structure where output data will be stored.
    @return Number of DLE-bytes in string
*/
uint16_t RFIDB1ProtocolInterface::GetDLEBytes(uint8_t *buff, uint16_t size)
{
    uint16_t i;
    uint16_t res = 0;

    for (i = 0; i < size; i++)
        if (buff[i] == B1Header_STX || buff[i] == B1Header_ETX || buff[i] == B1Header_DLE) res++;

    return res;
}

/**
    @brief  Function used to DLE-encode the data.
    @param[in/out]  DataSize - number of bytes placed in TxBuff before and after DLE encoding
*/
RFIDB1ProtocolStatusT RFIDB1ProtocolInterface::DLEEncodeData(uint16_t *DataSize)
{
    uint16_t InputIndex;
    uint16_t OutputIndex;
    uint16_t dleByles = this->GetDLEBytes(&this->TxBuff[1], *DataSize);
    uint8_t *ptr = &this->TxBuff[1];

    if (*DataSize + dleByles + 2 > this->TxBuffSizeMax)
        return RFIDB1ProtocolStatusT_TxDataOverflow;

    InputIndex = *DataSize;

    *DataSize = *DataSize + dleByles;

    OutputIndex = *DataSize - 1;

    do
    {
        InputIndex--;
        if (ptr[InputIndex] == B1Header_STX) {
            ptr[OutputIndex--] = B1Header_STX + B1Header_DLE;
            ptr[OutputIndex--] = B1Header_DLE;
        } else if (ptr[InputIndex] == B1Header_ETX) {
            ptr[OutputIndex--] = B1Header_ETX + B1Header_DLE;
            ptr[OutputIndex--] = B1Header_DLE;
        } else if (ptr[InputIndex] == B1Header_DLE) {
            ptr[OutputIndex--] = B1Header_DLE + B1Header_DLE;
            ptr[OutputIndex--] = B1Header_DLE;
        } else {
            ptr[OutputIndex--] = ptr[InputIndex];
        }
    } while (InputIndex > 0);

    return RFIDB1ProtocolStatusT_OK;
}

/**
    @brief  Function used to process incoming data byte by byte.
    @note   Function processed incoming data in compatibility with configuration in ApplicationVolatileConfiguration.
    @param[in] DataByte - next received byte of data.
    @return RFIDB1UARTDriverStatusT_InvalidPacket - received packet is invalid.
    @return RFIDB1UARTDriverStatusT_IncompletePacket - packet is not fully read.
    @return RFIDB1ProtocolStatusT_RxDataOverflow - in Rx packet is bigger than allocated Rx buff size
    @return RFIDB1UARTDriverStatusT_OK - correct packet received.
*/
RFIDB1ProtocolStatusT RFIDB1ProtocolInterface::ProcessDataByte(uint8_t DataByte) {
    bool PacketComplete = false;

    switch (this->HeaderType) {
        case HeaderTypeA:
            if (this->Receiving == false) {
                if (DataByte == B1Header_STX) {
                   // beginning of transmission, byte is discarted
                   this->Header[0] = DataByte;
                   this->Receiving = true;
                   this->ByteCounter = 0;
                }
            } else {
                switch (this->ByteCounter++) {
                    case 0: //Header->DataSize LSB
                        this->DataSize = DataByte;
                        this->Header[1] = DataByte;
                        break;

                    case 1: //Header->DataSize MSB
                        this->DataSize += ((uint16_t)DataByte) << 8;
                        this->Header[2] = DataByte;
                        if (this->DataSize == 0 || this->DataSize > this->RxBuffSizeMax) {

                            this->Receiving = false;
                            this->ByteCounter = 0;
                            return RFIDB1ProtocolStatusT_RxDataOverflow;
                        }
                        break;

                    case 2: //Header->CRC LSB
                        this->HeaderCRC = DataByte;
                        break;

                    case 3: //Header->CRC MSB
                        this->HeaderCRC += ((uint16_t)DataByte) << 8;

                        if (this->HeaderCRC != GetCCITTCRC(this->Header, 3)) {
                            return RFIDB1ProtocolStatusT_CRCError;
                        }
                         break;

                    default:
                        this->RxBuff[this->ByteCounter - 5] = DataByte;
                        if ((this->ByteCounter - 4) == this->DataSize) {
                            this->Receiving = false;
                            PacketComplete = true;
                            this->RxBuffSize = this->DataSize;
                            this->HeaderType = this->NewHeaderType;
                            if (this->HeaderType == HeaderTypeA)
                                this->HeaderSize = 5;
                            else
                                this->HeaderSize = 1;
                            this->ByteCounter = 0;
                            return RFIDB1ProtocolStatusT_OK;
                        }
                }
            }
            break;

        case HeaderTypeB:
            if (this->Receiving == false) {
                if (DataByte == B1Header_STX) {
                   // beginning of transmission, byte is discarted
                   this->Receiving = true;
                   this->ByteCounter = 0;
                }
            } else {
                if (DataByte == B1Header_ETX) {
                    // end of transmission
                    this->Receiving = false;
                    PacketComplete = true;

                    this->RxBuffSize = this->ByteCounter;
                    this->HeaderType = this->NewHeaderType;
                    if (this->HeaderType == HeaderTypeA)
                        this->HeaderSize = 5;
                    else
                        this->HeaderSize = 1;

                    this->ByteCounter = 0;
                    return RFIDB1ProtocolStatusT_OK;

                } else if (DataByte == B1Header_DLE) {
                    this->DLEReceived = true;
                } else {
                    if (this->DLEReceived) {
                        DataByte -= B1Header_DLE;
                        this->DLEReceived = false;
                    }
                    if (this->ByteCounter >= this->RxBuffSizeMax) {
                        this->Receiving = false;
                        this->ByteCounter = 0;
                        break;
                    }
                    if (this->ByteCounter >= this->RxBuffSizeMax)
                    {
                        this->Receiving = false;
                        return RFIDB1ProtocolStatusT_RxDataOverflow;
                    }

                    this->RxBuff[this->ByteCounter++] = DataByte;
                }
            }
            break;

        default:
            break;
    }
    if (PacketComplete == false) {
        return RFIDB1ProtocolStatusT_Incomplite;
    } else {
        return RFIDB1ProtocolStatusT_OK;
    }
}

/**
    @brief  Function used to Create outgoing packet using data already placed in TxBuff
    @param[in] HeaderType - next received byte of data.
    @param[in] DataSize - data size in TxBuff.
    @return RFIDB1ProtocolStatusT_InvalidParameters - wrong input parameters
    @return RFIDB1ProtocolStatusT_TxDataOverflow - if Tx buffer is to small to create packet
    @return RFIDB1UARTDriverStatusT_OK - correct packet received.
*/
RFIDB1ProtocolStatusT RFIDB1ProtocolInterface::CreateTxPacket(uint16_t DataSize)
{
    uint16_t CalculatedCRC = 0;
    RFIDB1ProtocolStatusT status;

    if (DataSize == 0)
        return RFIDB1ProtocolStatusT_InvalidParameters;

    switch (this->HeaderType) {
        case HeaderTypeA:
            if (DataSize + 5 > this->TxBuffSizeMax) {
                return RFIDB1ProtocolStatusT_TxDataOverflow;
            }
            this->TxBuff[0] = B1Header_STX;
            this->TxBuff[1] = (uint8_t)(DataSize & 0x00FF);
            this->TxBuff[2] = (uint8_t)((DataSize & 0xFF00) >> 8);

            CalculatedCRC = GetCCITTCRC(this->TxBuff, 3);
            this->TxBuff[3] = (uint8_t)((CalculatedCRC ) & 0x00FF);
            this->TxBuff[4] = (uint8_t)(((CalculatedCRC ) & 0xFF00) >> 8);

            DataSize += 5;

            this->TxBuffSize = DataSize;
            break;

        case HeaderTypeB:
            this->TxBuff[0] = B1Header_STX;
            status = this->DLEEncodeData(&DataSize);
            if (status!= RFIDB1ProtocolStatusT_OK)
                return status;
            DataSize++;//STX
            this->TxBuff[DataSize] = B1Header_ETX;
            DataSize++;//ETX
            this->TxBuffSize = DataSize;
            break;

        default:
            return RFIDB1ProtocolStatusT_SubModuleError;
    }

    return RFIDB1ProtocolStatusT_OK;
}

/**
	@brief Function gets data packet from UART and put in to RxBuff
	@param[out] DataSize - received data size
	@param[in] UartData - incoming data
	@param[in] UartDataSize - incoming data size
    @return RFIDB1ProtocolStatusT_OK.
    @return RFIDB1ProtocolStatusT_InvalidParameters - Null pointers received.
    @return RFIDB1ProtocolStatusT_RxDataOverflow - Data from Uart is greater then pointed DataSize.
    @return RFIDB1ProtocolStatusT_InvalidPacket - unexpected packet size.
    @return RFIDB1ProtocolStatusT_CRCError - CRC error in received data.
    @return RFIDB1ProtocolStatusT_NoRxPackets - No packets received.
	*/
RFIDB1ProtocolStatusT RFIDB1ProtocolInterface::GetB1Packet(uint16_t *DataSize, uint8_t *UartData, uint16_t *UartDataSize) {
    uint16_t CalculatedCRC = 0;
    uint16_t ReceivedCRC = 0;
    uint16_t k;
    uint8_t tmp[322];
    RFIDB1ProtocolStatusT status = RFIDB1ProtocolStatusT_Error;

    memset(tmp, 0x00, sizeof(tmp));

    if (DataSize == NULL || UartData == NULL || UartDataSize == NULL)
        return RFIDB1ProtocolStatusT_InvalidParameters;

    for (k = 0; k < *UartDataSize && status != RFIDB1ProtocolStatusT_OK && status != RFIDB1ProtocolStatusT_CRCError ; k++)
        status = ProcessDataByte(UartData[k]);

    memmove(UartData, &UartData[k], *UartDataSize - k);
    *UartDataSize -= k;

    if (status != RFIDB1ProtocolStatusT_OK)
        return status;

    *DataSize = this->RxBuffSize;

    switch (this->DataType) {

        //Plain
        case DataTypePlain:
            if (*DataSize < 3) { //Status + CRC
                return RFIDB1ProtocolStatusT_InvalidPacket;
            }
            ReceivedCRC = ((uint16_t)(this->RxBuff[*DataSize-2])) | ((uint16_t)(this->RxBuff[*DataSize-1]) << 8);
            CalculatedCRC = GetCCITTCRC(this->RxBuff, *DataSize-2);
            if (CalculatedCRC != ReceivedCRC)
                return RFIDB1ProtocolStatusT_CRCError;

            *DataSize -= 2; //Last Two bytes are CRC.
            return RFIDB1ProtocolStatusT_OK;

        //Encrypted + CRC
        case DataTypeEncrypted:
            if (!this->AesDecryptBuffer)
                return RFIDB1ProtocolStatusT_CommunicationParametersNotSet;
            //Size check. Must be multiple of 16.
            if (*DataSize % 16 != 0)
                return RFIDB1ProtocolStatusT_InvalidPacket;

            this->AesDecryptBuffer(this->RxBuff, (uint32_t)*DataSize, this->AESKey, this->AESInitializationVector);

            *DataSize = ((uint32_t)(this->RxBuff[1]) << 8) | ((uint32_t)(this->RxBuff[0])); //Two first Bytes are data size in encrypted packet.

            if (*DataSize < 5) //Size + Status + CRC
                return RFIDB1ProtocolStatusT_InvalidPacket;

            ReceivedCRC = ((uint16_t)(this->RxBuff[*DataSize-1]) << 8) | ((uint16_t)(this->RxBuff[*DataSize-2]));

            *DataSize -= 2; //Last Two bytes are CRC.
            CalculatedCRC = GetCCITTCRC(this->RxBuff, *DataSize);
            if (CalculatedCRC != ReceivedCRC)
                return RFIDB1ProtocolStatusT_CRCError;

            *DataSize -= 2; //First Two bytes are Data Size.

            return RFIDB1ProtocolStatusT_OK;
        default:
                break;
    }
    //We should never get here.
    return RFIDB1ProtocolStatusT_Error;
}

/**
	@brief Function prepares B1 packet based on current header type and packet encoding.
	@param[in] DataSize - data size
    @return RFIDB1ProtocolStatusT_OK.
    @return RFIDB1ProtocolStatusT_InvalidParameters - wrong input parameters.
    @return RFIDB1ProtocolStatusT_TxDataOverflow - Data from Uart is greater then pointed DataSize.
	*/
RFIDB1ProtocolStatusT RFIDB1ProtocolInterface::PutB1Packet(const uint16_t DataSize)
{
    uint16_t CalculatedCRC = 0;

    if (DataSize == 0)
        return RFIDB1ProtocolStatusT_InvalidParameters;

    switch (this->DataType) {
        case DataTypePlain:
           if (DataSize + this->HeaderSize + 1 > this->TxBuffSizeMax)
                return RFIDB1ProtocolStatusT_TxDataOverflow;

            CalculatedCRC = GetCCITTCRC(&this->TxBuff[this->HeaderSize], DataSize);
            this->TxBuff[this->HeaderSize + DataSize] = (uint8_t)((CalculatedCRC ) & 0x00FF);
            this->TxBuff[this->HeaderSize + DataSize + 1] = (uint8_t)(((CalculatedCRC ) & 0xFF00) >> 8);
            return CreateTxPacket(DataSize + 2);
        case DataTypeEncrypted:
            {
            uint16_t AESTxDataSize;
            uint8_t AESNullTailSize;

            if (!this->AesEncryptBuffer)
                return RFIDB1ProtocolStatusT_CommunicationParametersNotSet;

            AESNullTailSize = 16 - (DataSize + 4) % 16; //+2 for CRC bytes and + for AES data size
            if (AESNullTailSize == 16) {
                AESNullTailSize = 0;
            }
            AESTxDataSize = (( (DataSize+4)/16) * 16) + (( (DataSize+4) % 16) ? 16 : 0); //+2 for CRC bytes and + for AES data size

            if (AESTxDataSize + this->HeaderSize + 1 > this->TxBuffSizeMax)
                return RFIDB1ProtocolStatusT_TxDataOverflow;

            //Write DataSize at the beginning of packet.
            this->TxBuff[this->HeaderSize] = (uint8_t)((DataSize+4) & 0x00FF);
            this->TxBuff[this->HeaderSize+1] = (uint8_t)(((DataSize+4) & 0xFF00) >> 8);

            CalculatedCRC = GetCCITTCRC(&this->TxBuff[this->HeaderSize], DataSize+2);
            this->TxBuff[this->HeaderSize + DataSize + 2] = (uint8_t)((CalculatedCRC) & 0x00FF);
            this->TxBuff[this->HeaderSize + DataSize + 3] = (uint8_t)(((CalculatedCRC) & 0xFF00) >> 8);

            memset(&this->TxBuff[this->HeaderSize + DataSize + 4], 0, AESNullTailSize); //4 = 2 DataSize bytes + 2 CRC bytes. Null fill.

            this->AesEncryptBuffer(&this->TxBuff[this->HeaderSize], AESTxDataSize, this->AESKey, this->AESInitializationVector);

            //Send data to uart.
            return CreateTxPacket(AESTxDataSize);
            }
        default:
                return RFIDB1ProtocolStatusT_SubModuleError;
    }
}

/**
	@brief Function sets protocol header type
	@param[in] PacketHeaderType - new header type
    @return RFIDB1ProtocolStatusT_OK.
    @return RFIDB1ProtocolStatusT_InvalidParameters - wrong input parameters.
	*/
RFIDB1ProtocolStatusT RFIDB1ProtocolInterface::SetPacketHeaderType(HeaderTypeT PacketHeaderType)
{
    if (PacketHeaderType != HeaderTypeA && PacketHeaderType != HeaderTypeB)
        return RFIDB1ProtocolStatusT_InvalidParameters;

    this->NewHeaderType = PacketHeaderType;

    if (this->Receiving == false)
    {
        this->HeaderType = PacketHeaderType;
        if (PacketHeaderType == HeaderTypeA)
            this->HeaderSize = 5;
        else
            this->HeaderSize = 1;
    }

    return RFIDB1ProtocolStatusT_OK;
}


/**
	@brief Function sets encoding type
	@param[in] DataType - new header type
	@param[in] AESKey - AES key, optional, needed only if transmission is encrypted
	@param[in] AESInitializationVector - AES vector, optional, needed only if transmission is encrypted
    @return RFIDB1ProtocolStatusT_OK.
    @return RFIDB1ProtocolStatusT_InvalidParameters - wrong input parameters.
	*/
RFIDB1ProtocolStatusT RFIDB1ProtocolInterface::SetPacketEncoding(DataTypeT DataType, uint8_t *AESKey, uint8_t *AESInitializationVector)
{
    if (DataType == DataTypeEncrypted)
    {
        if (!AESKey || !AESInitializationVector || !this->AesDecryptBuffer || !this->AesEncryptBuffer)
            return RFIDB1ProtocolStatusT_InvalidParameters;

        memcpy(this->AESKey, AESKey, 16);
        memcpy(this->AESInitializationVector, AESInitializationVector, 16);
    }

    this->DataType = DataType;
    return RFIDB1ProtocolStatusT_OK;
}

/**
	@brief Function gets pointer to Tx data buff
    @return data pointer
	*/
uint8_t * RFIDB1ProtocolInterface::GetTxDataPointer()
{
    if (this->DataType == DataTypePlain)
        return &this->TxBuff[this->HeaderSize];
    else
        return &this->TxBuff[this->HeaderSize + 2];
}

uint8_t RFIDB1ProtocolInterface::GetHeaderSize()
{
    return this->HeaderSize;
}

uint16_t RFIDB1ProtocolInterface::GetTxBuffSizeMax()
{
    return this->TxBuffSizeMax;
}

uint16_t RFIDB1ProtocolInterface::GetTxBuffSize()
{
    return this->TxBuffSize;
}

void RFIDB1ProtocolInterface::SetTxBuffSize(uint16_t value)
{
    this->TxBuffSize = value;
}

/**
	@brief Function gets pointer to Rx data buff
    @return data pointer
    */
uint8_t * RFIDB1ProtocolInterface::GetRxDataPointer()
{
    if (this->DataType == DataTypePlain)
        return &this->RxBuff[0];
    else
        return &this->RxBuff[2];
}

uint8_t * RFIDB1ProtocolInterface::GetTxBuff()
{
    return this->TxBuff;
}

uint8_t * RFIDB1ProtocolInterface::GetRxBuff()
{
    return this->RxBuff;
}
