/*
  RFIDB1ClientProtocol.h - this file is part of the RFID-B1 
  library for RFIDB1 modules by Eccel Technology Ltd.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

   Authors:  Marcin Baliniak, Piotr Szmelcer, Damian Gowor

   Version: 1.0

  See file LICENSE.txt for further informations on licensing terms.
*/

#ifndef PROTOCOL_H_INCLUDED
#define PROTOCOL_H_INCLUDED

#include <stdbool.h>
#include <stdint.h>

#define B1Header_DLE (0x10)
#define B1Header_ETX (0x03)
#define B1Header_STX (0x02)

typedef enum {
    RFIDB1ProtocolStatusT_OK = 0,
    RFIDB1PotocolStatusT_NotInitialized,
    RFIDB1ProtocolStatusT_RxDataOverflow,
    RFIDB1ProtocolStatusT_TxDataOverflow,
    RFIDB1ProtocolStatusT_Timeout,
    RFIDB1ProtocolStatusT_InvalidPacket,
    RFIDB1ProtocolStatusT_Incomplite,
    RFIDB1ProtocolStatusT_CRCError,
    RFIDB1ProtocolStatusT_NoRxPackets,
    RFIDB1ProtocolStatusT_InvalidParameters,
    RFIDB1ProtocolStatusT_UartError,
    RFIDB1ProtocolStatusT_Error,
    RFIDB1ProtocolStatusT_MemoryAllocationError,
    RFIDB1ProtocolStatusT_CommunicationParametersNotSet,
    RFIDB1ProtocolStatusT_SubModuleError
} RFIDB1ProtocolStatusT;

typedef enum {
    DataTypePlain = 0,
    DataTypeEncrypted = 1
} DataTypeT;

typedef enum {
    HeaderTypeA = 0,
    HeaderTypeB = 1
}HeaderTypeT;

typedef struct {
	uint8_t *InputBuffer;		/*!< Pointer to an input buffer which will be used by the driver. */
	uint8_t *OutputBuffer;		/*!< Pointer to an output buffer which will be used by the driver. */
	uint16_t InputBufferSize;	/*!< Size of the input buffer. */
	uint16_t OutputBufferSize;	/*!< Size of the output buffer. */
    void (*AesEncryptBuffer)(uint8_t* buff, uint32_t length, const uint8_t* key, const uint8_t* iv); /*!< Pointer to function for AES encrypt */
    void (*AesDecryptBuffer)(uint8_t* buff, uint32_t length, const uint8_t* key, const uint8_t* iv); /*!< Pointer to function for AES decrypt */
} RFIDB1_IProtocolConfigurationT;

class RFIDB1ProtocolInterface
{
    public:
        RFIDB1ProtocolInterface();
        RFIDB1ProtocolStatusT Initialise(RFIDB1_IProtocolConfigurationT *config);
        RFIDB1ProtocolStatusT GetB1Packet(uint16_t *DataSize, uint8_t *UartData, uint16_t *UartDataSize);
        RFIDB1ProtocolStatusT PutB1Packet(const uint16_t DataSize);
        RFIDB1ProtocolStatusT SetPacketHeaderType(HeaderTypeT PacketHeaderType);
        RFIDB1ProtocolStatusT SetPacketEncoding(DataTypeT DataType, uint8_t *AESKey, uint8_t *AESInitializationVector);
        uint8_t * GetTxDataPointer();
        uint8_t * GetRxDataPointer();
        uint8_t * GetTxBuff();
        uint8_t * GetRxBuff();
        uint8_t GetHeaderSize();
        uint16_t GetTxBuffSizeMax();
        uint16_t GetTxBuffSize();
        void SetTxBuffSize(uint16_t);
    private:
        uint16_t GetDLEBytes(uint8_t *buff, uint16_t size);
        RFIDB1ProtocolStatusT DLEEncodeData(uint16_t *DataSize);
        RFIDB1ProtocolStatusT ProcessDataByte(uint8_t DataByte);
        RFIDB1ProtocolStatusT CreateTxPacket(uint16_t DataSize);

        uint8_t  AESInitializationVector[16];
        uint8_t  AESKey[16];
        void (*AesEncryptBuffer)(uint8_t* buff, uint32_t length, const uint8_t* key, const uint8_t* iv);
        void (*AesDecryptBuffer)(uint8_t* buff, uint32_t length, const uint8_t* key, const uint8_t* iv);
        uint8_t *RxBuff;
        uint8_t *TxBuff;
        uint8_t Header[3];
        uint16_t RxBuffSize;
        uint16_t RxBuffSizeMax;
        uint16_t TxBuffSize;
        uint16_t TxBuffSizeMax;
        uint16_t ByteCounter;
        uint16_t DataSize;
        uint16_t HeaderCRC;
        bool Receiving;
        DataTypeT  DataType;
        HeaderTypeT  HeaderType, NewHeaderType;
        uint8_t HeaderSize;
        uint8_t DLEReceived;
};

#endif // PROTOCOL_H_INCLUDED
