/*
  RFIDB1ClientInterface.cpp - this file is part of the RFID-B1
  library for RFIDB1 modules by Eccel Technology Ltd.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

   Authors:  Marcin Baliniak, Piotr Szmelcer, Damian Gowor

   Version: 1.0

  See file LICENSE.txt for further informations on licensing terms.
*/


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>

#include "RFIDB1ClientInterface.h"
#include "RFIDB1SimpleInterface.h"

Stream *serialPort;

uint8_t ACK_frame[1024];
uint16_t ACK_frame_len;
bool ACK_received;
bool ASYNC_received;
uint8_t ASYNC_byte;

static void handleRequest(uint8_t *data, uint16_t size)
{
  serialPort->write(data, size);
}

static void handleResponse(uint8_t *data, uint16_t size)
{
  if (data[0] == B1ResponseACK)
  {
    ACK_frame_len = size-1;
    ACK_received = true;
    memcpy(ACK_frame, &data[1], size -1);
  }
  else if (data[0] == B1ResponseAsyncPacket)
  {
    ASYNC_byte = data[1];
    ASYNC_received = true;
  }
}

/**
    @brief  Function waits for answer from module
    @param[in] needAck - function should wait for ACK frame
    @param[in] needAsync - function should wait for Async frame
    @param[in] timeout - timeout in ms
    @return true - requested frames received, false - timeout
*/
bool RFIDB1SimpleInterface::wait4Answer(bool needAck, bool needAsync, uint16_t timeout)
{
    uint8_t buff[1024];
    uint32_t tout = millis() + timeout;
    int lenght;

    ACK_frame_len = 0;
    ACK_received = false;
    ASYNC_received = false;

    while (millis() < tout)
    {

      lenght = serialPort->available();
      if (lenght > 0)
      {
        if (lenght > sizeof(buff))
          lenght = sizeof(buff);

        lenght = serialPort->readBytes(buff, lenght);
        if (lenght > 0)
          B1ClientInterface->ParseIncomingData(buff, lenght);
      }

      if (ASYNC_received == needAsync && ACK_received == needAck)
        return true;
    }

    return false;
}

/**
    @brief Constructor - used to Initialise B1Client interface for default settings
    @param[in] serial - serial used for B1 communication
*/
RFIDB1SimpleInterface::RFIDB1SimpleInterface(Stream &serial)
{
    serialPort = &serial;

    RFIDB1_InterfaceConfigurationT config;
    B1ClientInterface = new RFIDB1_Interface();

    memset(&config, 0, sizeof(config));

    config.handleRequest = handleRequest;
    config.handleResponse = handleResponse;
    config.InputBuffer = rxBuff;
    config.InputBufferSize = sizeof(rxBuff);
    config.OutputBuffer = txBuff;
    config.OutputBufferSize = sizeof(txBuff);

    if (B1ClientInterface->Initialise(&config) != B1StatusT_OK)
        return;

    B1ClientInterface->SetPacketHeaderType(HeaderTypeA);
    B1ClientInterface->SetPacketEncoding(DataTypePlain, NULL, NULL);
}

/**
    @brief  Function gets UID lenght
    @return UID lenght
*/
uint8_t RFIDB1SimpleInterface::GetUIDLen(void)
{
    return uidLen;
}

/**
    @brief  Function gets UID type
    @return UID type
*/
uint8_t RFIDB1SimpleInterface::GetUIDType(void)
{
    return uidType;
}

/**
    @brief  Function gets UID bytes
    @return UID bytes
*/
uint8_t * RFIDB1SimpleInterface::GetUIDBytes(void)
{
    return uidBytes;
}

/**
    @brief  Function check if tag is present
    @return true if present and false if not
*/
const char * RFIDB1SimpleInterface::GetUidTypeString(void)
{
  switch (uidType)
  {
  case TagType_NoTag:             return "NoTag";
  case TagType_NotComplete:       return "NotComplete";
  case TagType_Ultralight:        return "Ultralight";
  case TagType_UltralightEV1_80B: return "UltralightEV1_80B";
  case TagType_UltralightEV1_164B:return "UltralightEV1_164B";
  case TagType_ClassicMini:       return "ClassicMini";
  case TagType_Classic1K:         return "Classic1K";
  case TagType_Classic4K:         return "Classic4K";
  case TagType_NTAG203F:          return "NTAG203F";
  case TagType_NTAG210:           return "NTAG210";
  case TagType_NTAG212:           return "NTAG212";
  case TagType_NTAG213F:          return "NTAG213F";
  case TagType_NTAG216F:          return "NTAG216F";
  case TagType_NTAG213:           return "NTAG213";
  case TagType_NTAG215:           return "NTAG215";
  case TagType_NTAG216:           return "NTAG216";
  default:
      return "Unknown";
      break;
  }
}

/**
    @brief  Function check if tag is present
    @return true if present and false if not
*/
bool RFIDB1SimpleInterface::isTagPresent(void)
{
  uint8_t command;

  uidType = TagType_NoTag;
  uidLen = 0;
  command = B1RFIDCommandGetUIDAndType;

  B1ClientInterface->SendWriteToRFIDMemoryCommand(B1RegisterCommand, &command, 1);

  if (wait4Answer(true, true, 200) && (ASYNC_byte & B1AsyncPacketParamBit_RFIDCommand))
  {
    B1ClientInterface->SendReadFromRFIDMemoryCommand(B1RegisterTagUID, 12);

    if (wait4Answer(true, false, 200))
    {
      uidType = ACK_frame[10];
      uidLen = ACK_frame[11];
      memcpy(uidBytes, ACK_frame, uidLen);
    }
  }

  return (TagType_NoTag != uidType);
}

/**
    @brief  Function stops RF communication
    @return true if success
*/
bool RFIDB1SimpleInterface::Halt(void)
{
  uint8_t command;

  command = B1RFIDCommandHalt;

  B1ClientInterface->SendWriteToRFIDMemoryCommand(B1RegisterCommand, &command, 1);

  return wait4Answer(true, true, 200);
}

/**
    @brief Function reads from module memory
    @param[in] address - memory address
    @param[in] len - length to read
    @return true - Success, false - Error
*/
bool RFIDB1SimpleInterface::ReadFromRfidMemory(uint16_t address, uint16_t len)
{
  B1ClientInterface->SendReadFromRFIDMemoryCommand(address, len);
  if (!wait4Answer(true, false, 200))
  {
    printf("ReadFromRFIDMemory failed: no ACK\n");
    return false;
  }

  return true;
}

/**
    @brief Function writes to module memory
    @param[in] address - memory address
    @param[in] buff - pointer with data to write
    @param[in] len - number of bytes to write
    @return 0 - Success, -1 - Error
*/
bool RFIDB1SimpleInterface::WriteToRfidMemory(uint16_t address, uint8_t *buff, uint16_t len)
{
  B1ClientInterface->SendWriteToRFIDMemoryCommand(address, buff, len);

  if (!wait4Answer(true, false, 200))
    return false;

  B1ClientInterface->SendReadFromRFIDMemoryCommand(address, len);
  if (!wait4Answer(true, false, 200))
    return false;

  if (memcmp(buff, ACK_frame, len))
    return false;

  return true;
}

/**
    @brief  Function transform status from numeric to string
    @param[in] status - status in numeric format
    @return pointer to null terminated string
*/
const char* RFIDB1SimpleInterface::GetResultString(uint8_t status)
{
  switch (status)
  {
    case 0x00:  return "No ERROR";
    case 0x01:  return "Invalid command";
    case 0x02:  return "Invalid command parameter";
    case 0x03:  return "Indexes out of range";
    case 0x04:  return "Error when writing to non volatile memory";
    case 0x05:  return "System error";
    case 0x06:  return "Tag CRC error";
    case 0x07:  return "Tag collision";
    case 0x08:  return "Tag is not present";
    case 0x09:  return "Tag authentication error";
    case 0x0A:  return "Tag value block corrupted";
    case 0x0B:  return "Module overheated";
    case 0x0C:  return "Tag not supported";
    case 0x0D:  return "Tag communication error";
    case 0x0E:  return "Invalid password";
    case 0x0F:  return "Already locked";
    case 0xFF:  return "Module busy";
    default:  return "Unknown";
  }
}

//------------------------------ RFID functions --------------------------------------------
/**
    @brief Function used to read blocks from Mifare tag
    @param[in] blockNum - start block
    @param[in] blocks2read - number of blocks to read
    @param[in] keyAB - type of key A/B
    @param[in] key - key used to read memory
    @param[in] dataOnly - skip special blocks
    @return true - Success, false - Error
*/
bool RFIDB1SimpleInterface::ReadBlock(int blockNum, int blocks2read, char keyAB, uint8_t *key, bool dataOnly, uint8_t *outBuff)
{
  uint8_t cmd[30];

  if (blocks2read * 16 > 256)
  {
    //printf("Trying to read too much data\n");
    outBuff[0] = 1;
    return false;
  }

  if (keyAB != 'A' && keyAB != 'B')
  {
    Serial.printf("Please specify key A or B\n");
    outBuff[0] = 2;
    return false;
  }

  if (dataOnly)
    cmd[0] = B1RFIDCommandReadDataBlock;
  else
    cmd[0] = B1RFIDCommandReadBlock;

  cmd[1] = blockNum;
  cmd[2] = blocks2read;
  cmd[3] = 0;
  cmd[4] = 1 << 6;
  memcpy(&cmd[5], key, 6);

  if (keyAB == 'B')
    cmd[4] |= 1 << 7;

  B1ClientInterface->SendWriteToRFIDMemoryCommand(B1RegisterCommand, cmd, 11);

  if (wait4Answer(true, true, 200) && (ASYNC_byte & B1AsyncPacketParamBit_RFIDCommand))
  {
    //read the status
    B1ClientInterface->SendReadFromRFIDMemoryCommand(B1RegisterResult, 1);
    if (!wait4Answer(true, false, 200))
      return false;

    if (ACK_frame[0] != 0x00)
    {
      Serial.printf("Read error: %s\n", GetResultString(ACK_frame[0]));
      outBuff[0] = 3;
      outBuff[1] = ACK_frame[0];
      return false;
    }
  }
  else
  {

    if (!ACK_received)
    {
      Serial.printf("No ACK for read command\n");
      outBuff[0] = 4;
    }

    if (!ASYNC_received)
    {
      outBuff[0] = 5;
      Serial.printf("No ASYNC for read block command\n");
    }


    return false;
  }

  if (!ReadFromRfidMemory(32, blocks2read * 16))
  {
    Serial.printf("ReadFromRfidMemory error\n");
    outBuff[0] = 6;
    return false;
  }

  memcpy(outBuff, ACK_frame, 16 * blocks2read);

  return true;
}


/**
    @brief Function used to write blocks to Mifare tag
    @param[in] cblock_num - start block
    @param[in] cblocks2read - number of blocks to read
    @param[in] cdataAddr - address to store read data
    @param[in] ckeyAB - type of key A/B
    @param[in] ckey - key used to read memory
    @param[in] dataOnly - skip special blocks
    @return 0 - Success, -1 - Error
*/
bool RFIDB1SimpleInterface::WriteBlock(int blockNum, int blocks2write, char keyAB, uint8_t *key, bool dataOnly, uint8_t *buff2write)
{
  uint8_t cmd[30];

  WriteToRfidMemory(32, buff2write, 16 * blocks2write);

  if (blocks2write * 16 > 256)
  {
    Serial.printf("You trying to write out of data buffer\n");
    return false;
  }

  if (keyAB != 'A' && keyAB != 'B')
  {
    Serial.printf("Please provide correct key A or B\n");
    return false;
  }

  if (dataOnly)
    cmd[0] = B1RFIDCommandWriteDataBlock;
  else
    cmd[0] = B1RFIDCommandWriteBlock;

  cmd[1] = blockNum;
  cmd[2] = blocks2write;
  cmd[3] = 0;
  cmd[4] = 1 << 6;
  memcpy(&cmd[5], key, 6);

  if (keyAB == 'B')
    cmd[4] |= 1 << 7;

  B1ClientInterface->SendWriteToRFIDMemoryCommand(B1RegisterCommand, cmd, 11);

  if (wait4Answer(true, true, 200) && (ASYNC_byte & B1AsyncPacketParamBit_RFIDCommand))
  {
    //read the status
    B1ClientInterface->SendReadFromRFIDMemoryCommand(B1RegisterResult, 1);
    if (!wait4Answer(true, false, 200))
    {
      Serial.printf("B1RFIDCommandWriteBlock failed: no ACK\n");
      return false;
    }

    if (ACK_frame[0] != 0x00)
    {
      Serial.printf("Write error: %s\n", GetResultString(ACK_frame[0]));
      return false;
    }
  }
  else
  {
    if (!ACK_received)
      Serial.printf("No ACK for write command\n");

    if (!ASYNC_received)
      Serial.printf("No ASYNC for write block command\n");

    return false;
  }

  return true;
}

