/**
 *
 * @file      RFIDB1ClientProtocol.c
 * @brief     Protocol for RFIDB1 interface
 * @author    Marcin Baliniak, Piotr Szmelcer
 * @date      26/03/2019
 * @version   1.1
 * @copyright Eccel Technology Ltd
 */

#ifndef RFIDB1LIBRARY_H_INCLUDED
#define RFIDB1LIBRARY_H_INCLUDED

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include "RFIDB1ClientProtocol.h"

#define B1TimeoutMs_Startup    (100)
#define B1TimeoutMs_Interrupt  (50)
#define B1TimeoutMs_Comparator (50)
#define B1TimeoutMs_CommunicationParameterChange  (50)

//Tag types.
#define TagType_NoTag               (0x00)
#define TagType_NotComplete         (0x01)
#define TagType_Ultralight          (0x02)
#define TagType_UltralightEV1_80B   (0x03)
#define TagType_UltralightEV1_164B  (0x04)
#define TagType_ClassicMini         (0x05)
#define TagType_Classic1K           (0x06)
#define TagType_Classic4K           (0x07)
#define TagType_NTAG203F            (0x08)
#define TagType_NTAG210             (0x09)
#define TagType_NTAG212             (0x0A)
#define TagType_NTAG213F            (0x0B)
#define TagType_NTAG216F            (0x0C)
#define TagType_NTAG213             (0x0D)
#define TagType_NTAG215             (0x0E)
#define TagType_NTAG216             (0x0F)
#define TagType_Unknown             (0x10)

typedef enum {
    //RFID driver status
    B1StatusT_OK = 0x00,
    B1StatusT_RxDataOverflow,
    B1StatusT_TxDataOverflow,
    B1StatusT_CommunicationTimeout,
    B1StatusT_InvalidPacket,
    B1StatusT_NoRxPackets,
    B1StatusT_InvalidParameters,
    B1StatusT_Error,
    B1StatusT_SubModuleError,
    B1StatusT_UnexpectedPacket,
    B1StatusT_NotInitialized, //10
    B1StatusT_Timeout,
} RFIDB1_StatusT;

/// <summary>
/// Enumeration respresenting the possible responses from B1 module as described in User Manual. Last one 0xFE is not directly send from B1 module.
/// They are generated by this class when invalid packet is received or when timeout is generater per the specification.
/// </summary>
typedef enum {
    B1ResponseACK = 0,
    B1ResponseInvalidCommand = 0x01,
    B1ResponseInvalidParameter = 0x02,
    B1ResponseProtocolError = 0x03,
    B1ResponseMemoryError = 0x04,
    B1ResponseSystemError = 0x05,
    B1ResponseModuleTimeout = 0x06,
    B1ResponseOverflow = 0x07,
    B1ResponseAsyncPacket = 0x08,
    B1ResponseBusy = 0x09,
    B1ResponseSystemStart = 0x0A,

    B1ResponseInvalidPacket = 0xFE,
} B1ResponseT;

#define B1AsyncPacketParamBit_IO0Edge     (0x01 << 0)
#define B1AsyncPacketParamBit_IO1Edge     (0x01 << 1)
#define B1AsyncPacketParamBit_IO2Edge     (0x01 << 2)
#define B1AsyncPacketParamBit_IO3Edge     (0x01 << 3)
#define B1AsyncPacketParamBit_Comparator  (0x01 << 4)
#define B1AsyncPacketParamBit_RFIDCommand (0x01 << 5)

typedef enum {
    B1RequestCommandDummy = 0x00,
    B1RequestCommandWriteToRFIDMemory,
    B1RequestCommandReadFromRFIDMemory,
    B1RequestCommandEnterSleepMode,
    B1RequestCommandReset,
    B1RequestCommandSetBaudRate,
    B1RequestCommandSetDataType,
    B1RequestCommandSetHeaderType,
    B1RequestCommandSetIOState,
    B1RequestCommandReadIOState,
    B1RequestCommandSetIOInterrupt,
    B1RequestCommandMeasureVoltage,
    B1RequestCommandMeasureDieTemperature,
    B1RequestCommandSetIDACCurrent,
    B1RequestCommandEnableComparator,
    B1RequestCommandDisableComparator,
    B1RequestCommandEnablePWM,
    B1RequestCommandSetAESInitVector,
    B1RequestCommandSetAESKey,
    B1RequestCommandReadAESInitVector,
    B1RequestCommandReadAESKey,
} B1RequestCommandT;

typedef enum {
    B1RFIDCommandGetUIDAndType = 0x01,
    B1RFIDCommandReadBlock,//
    B1RFIDCommandWriteBlock,//
    B1RFIDCommandReadDataBlock,//
    B1RFIDCommandWriteDataBlock,//
    B1RFIDCommandReadPage,//
    B1RFIDCommandWritePage,//
    B1RFIDCommandEncryptData,//
    B1RFIDCommandDecryptData,//
    B1RFIDCommandReadValue,//
    B1RFIDCommandWriteValue,//
    B1RFIDCommandIncrementValue,//
    B1RFIDCommandDecrementValue,//
    B1RFIDCommandRestoreValue,//
    B1RFIDCommandTransferValue,//
    B1RFIDCommandRecoverValue,//
    B1RFIDCommandGetVersion,//
    B1RFIDCommandReadSignature,//
    B1RFIDCommandConfigureUID,//
    B1RFIDCommandReadCounter,//
    B1RFIDCommandIncrementCounter,//
    B1RFIDCommandCheckTearingEvent,//
    B1RFIDCommandPasswordAuthentication,
    B1RFIDCommandHalt,//
    B1RFIDCommandCalculateCRC,//
    B1RFIDCommandCopyData,
    B1RFIDCommandUnlock,//
    B1RFIDCommandLock,//
    B1RFIDCommandGetModuleVersion,
    B1RFIDCommandResetToFactoryDefaults,
    B1RFIDCommandEnumerateTagsUID,
    B1RFIDCommandEnumerateTagsUIDandType,
    B1RFIDCommandSelectTag,
    B1RFIDCommandPolling,
} B1RfidCommandT;

typedef enum {
    B1RegisterResult                 = (0x0000),
    B1RegisterCommand                = (0x0001),
    B1RegisterCommandParameters      = (0x0002),
    B1RegisterTagUID                 = (0x0014),
    B1RegisterTagType                = (0x001E),
    B1RegisterTagUIDSize             = (0x001F),
    B1RegisterDataBuffer             = (0x0020),
    B1RegisterPassword               = (0x0120),
    B1RegisterAESInitVector0         = (0x0128),
    B1RegisterAESInitVector1         = (0x0138),
    B1RegisterAESKey0                = (0x0148),
    B1RegisterAESKey1                = (0x0158),
    B1RegisterAuthKeyPass00          = (0x0168),
    B1RegisterAuthKeyPass01          = (0x016E),
    B1RegisterAuthKeyPass02          = (0x0174),
    B1RegisterAuthKeyPass03          = (0x017A),
    B1RegisterAuthKeyPass04          = (0x0180),
    B1RegisterAuthKeyPass05          = (0x0186),
    B1RegisterAuthKeyPass06          = (0x018C),
    B1RegisterAuthKeyPass07          = (0x0192),
    B1RegisterAuthKeyPass08          = (0x0198),
    B1RegisterAuthKeyPass09          = (0x019E),
    B1RegisterAuthKeyPass10          = (0x01A4),
    B1RegisterAuthKeyPass11          = (0x01AA),
    B1RegisterAuthKeyPass12          = (0x01B0),
    B1RegisterAuthKeyPass13          = (0x01B6),
    B1RegisterAuthKeyPass14          = (0x01BC),
    B1RegisterAuthKeyPass15          = (0x01C2),
    B1RegisterAuthKeyPass16          = (0x01C8),
    B1RegisterAuthKeyPass17          = (0x01CE),
    B1RegisterAuthKeyPass18          = (0x01D4),
    B1RegisterAuthKeyPass19          = (0x01DA),
    B1RegisterAuthKeyPass20          = (0x01E0),
    B1RegisterAuthKeyPass21          = (0x01E6),
    B1RegisterAuthKeyPass22          = (0x01EC),
    B1RegisterAuthKeyPass23          = (0x01F2),
    B1RegisterAuthKeyPass24          = (0x01F8),
    B1RegisterAuthKeyPass25          = (0x01FE),
    B1RegisterAuthKeyPass26          = (0x0204),
    B1RegisterAuthKeyPass27          = (0x020A),
    B1RegisterAuthKeyPass28          = (0x0210),
    B1RegisterAuthKeyPass29          = (0x0216),
    B1RegisterAuthKeyPass30          = (0x021C),
    B1RegisterAuthKeyPass31          = (0x0222),
    B1RegisterAuthKeyPass32          = (0x0228),
    B1RegisterAuthKeyPass33          = (0x022E),
    B1RegisterAuthKeyPass34          = (0x0234),
    B1RegisterAuthKeyPass35          = (0x023A),
    B1RegisterAuthKeyPass36          = (0x0240),
    B1RegisterAuthKeyPass37          = (0x0246),
    B1RegisterAuthKeyPass38          = (0x024C),
    B1RegisterAuthKeyPass39          = (0x0252),
    B1RegisterUserMemory             = (0x0258),
} B1RegisterT;

typedef enum {
    Gpio0 = 0,
    Gpio1 = 1,
    Gpio2 = 2,
    Gpio3 = 3
} GpioT;

typedef enum {
    GpioStateDisabledHiZ = 0,
    GpioStateInput = 1,
    GpioStateOutputLow = 2,
    GpioStateOutputHigh = 3
} GpioStateT;

typedef enum {
    GpioInterruptFallingEdge = 0,
    GpioInterruptRisingEdge = 1,
    GpioInterruptAnyEdge = 2,
    GpioInterruptDisabled = 3
} GpioInterruptT;

typedef enum {
    AdcSourceAdcPin = 0,
    AdcSourcePowerSupply = 1
} AdcSourceT;

typedef enum {
    VoltageFormatUint32_mv = 0,
    VoltageFormatFloat_mV = 1,
    VoltageFormatFloat_V = 2
} VoltageFormatT;

typedef enum {
    CurrentFormatInt32_nA = 0,
    CurrentFormatFloat_nA = 1,
    CurrentFormatFloat_uA = 2
} CurrentFormatT;

typedef enum {
    PWMConfigFormatUint32_Hz = 0,
    PWMConfigFormatFloat_Hz = 1,
    PWMConfigFormatUint32_us = 2,
    PWMConfigFormatFloat_s = 3
} PWMConfigFormatT;

typedef enum {
    TemperatureFormatInt32_mC = 0,
    TemperatureFormatFloat_mC = 1,
    TemperatureFormatFloat_C = 2,
    TemperatureFormatInt32_mF = 3,
    TemperatureFormatFloat_mF = 4,
    TemperatureFormatFloat_F = 5
} TemperatureFormatT;

typedef enum {
    ComparatorReferenceVoltageRef1V25 = 0,
    ComparatorReferenceVoltageRef2V5 = 1,
    ComparatorReferenceVoltageRefDivVdd = 2
} ComparatorReferenceVoltageT;

typedef enum {
    ComparatorOutputPinDisabled = 0,
    ComparatorOutputPinEnabled = 1,
    ComparatorOutputPinEnabledNegated = 2
} ComparatorOutputPinT;

typedef enum {
    ComparatorAsyncPacketEdgeSensitivityDisabled = 0,
    ComparatorAsyncPacketEdgeSensitivityFallingEdge = 1,
    ComparatorAsyncPacketEdgeSensitivityRisingEdge = 2,
    ComparatorAsyncPacketEdgeSensitivityAnyEdge = 3
} ComparatorAsyncPacketEdgeSensitivityT;

typedef struct RFIDB1_InterfaceT RFIDB1_InterfaceT;
typedef struct RFIDB1_ObjectT RFIDB1_ObjectT;

struct RFIDB1_ObjectT
{
    Framework_RFIDB1ProtocolObjectT ProtocolObject;
    Framework_RFIDB1ProtocolInterfaceT ProtocolInterface;

    //functions used to Enable,Disable, Reset module, not needed for UART communication
    int (*getNSleep)(RFIDB1_ObjectT*);
    void (*setResetPin)(RFIDB1_ObjectT*, int);
    void (*setPowerPin)(RFIDB1_ObjectT*, int);
    void (*delayMs)(RFIDB1_ObjectT*, int);

    void (*handleResponse)(RFIDB1_ObjectT*, uint8_t *data, uint16_t size);
    void (*handleRequest)(RFIDB1_ObjectT*, uint8_t *data, uint16_t size);
    void *UserData;                                 //pointer for user data
};

typedef struct {
    void (*handleResponse)(RFIDB1_ObjectT*, uint8_t *data, uint16_t size);
    void (*handleRequest)(RFIDB1_ObjectT*, uint8_t *data, uint16_t size);

    //optional parameters, needed if user wan to use functions EnableModule, DisableModule, HardResetModule, ResetToDefaults
    int (*getNSleep)(RFIDB1_ObjectT*);
    void (*setResetPin)(RFIDB1_ObjectT*, int);
    void (*setPowerPin)(RFIDB1_ObjectT*, int);
    void (*delayMs)(RFIDB1_ObjectT*, int);
    void (*AesEncryptBuffer)(uint8_t* buff, uint32_t length, const uint8_t* key, const uint8_t* iv); /*!< Pointer to function for AES encrypt */
    void (*AesDecryptBuffer)(uint8_t* buff, uint32_t length, const uint8_t* key, const uint8_t* iv); /*!< Pointer to function for AES decrypt */

	uint8_t *InputBuffer;			/*!< Pointer to an input buffer which will be used by the driver. */
	uint8_t *OutputBuffer;		/*!< Pointer to an output buffer which will be used by the driver. */
	uint16_t InputBufferSize;	/*!< Size of the input buffer. */
	uint16_t OutputBufferSize;	/*!< Size of the output buffer. */
} RFIDB1_InterfaceConfigurationT;

struct RFIDB1_InterfaceT {
    //external
    RFIDB1_StatusT (*Initialise)(RFIDB1_ObjectT*, RFIDB1_InterfaceConfigurationT *);
    void (*SetUserData)(RFIDB1_ObjectT*, void *data);
    void* (*GetUserData)(RFIDB1_ObjectT*);

    RFIDB1_StatusT (*DisableModule)(RFIDB1_ObjectT*);
    RFIDB1_StatusT (*EnableModule)(RFIDB1_ObjectT*);
    RFIDB1_StatusT (*HardResetModule)(RFIDB1_ObjectT*);
    RFIDB1_StatusT (*ResetToDefaults)(RFIDB1_ObjectT*);

    RFIDB1_StatusT (*SetPacketHeaderType)(RFIDB1_ObjectT*, HeaderTypeT PacketHeaderType);
    RFIDB1_StatusT (*SetPacketEncoding)(RFIDB1_ObjectT*, uint8_t PacketEncoding, uint8_t *AESKey, uint8_t *AESInitializationVector);

    RFIDB1_StatusT (*ParseIncomingData)(RFIDB1_ObjectT*, uint8_t *data, uint16_t size);
    //UART Commands:
    RFIDB1_StatusT (*SendDummyCommand)(RFIDB1_ObjectT* );
    RFIDB1_StatusT (*SendWriteToRFIDMemoryCommand)(RFIDB1_ObjectT* , const uint16_t Address, const uint8_t *TxData, const uint16_t TxDataSize);
    RFIDB1_StatusT (*SendReadFromRFIDMemoryCommand)(RFIDB1_ObjectT* , const uint16_t Address, uint16_t BytesToRead);
    RFIDB1_StatusT (*SendEnterSleepModeCommand)(RFIDB1_ObjectT*);
    RFIDB1_StatusT (*SendSoftwareResetCommand)(RFIDB1_ObjectT*);
    RFIDB1_StatusT (*SendSetBaudRateCommand)(RFIDB1_ObjectT*, uint32_t BaudRate);
    RFIDB1_StatusT (*SendSetDataTypeCommand)(RFIDB1_ObjectT*, DataTypeT DataType);
    RFIDB1_StatusT (*SendSetHeaderTypeCommand)(RFIDB1_ObjectT*, HeaderTypeT HeaderType);
    RFIDB1_StatusT (*SendSetIOStateCommand)(RFIDB1_ObjectT*, const GpioT Gpio, const GpioStateT GpioState);
    RFIDB1_StatusT (*SendReadIOStateCommand)(RFIDB1_ObjectT*, const GpioT Gpio);
    RFIDB1_StatusT (*SendSetIOInterruptCommand)(RFIDB1_ObjectT*, const GpioT Gpio, const GpioInterruptT GpioInterrupt);
    RFIDB1_StatusT (*SendMeasureVoltageCommand)(RFIDB1_ObjectT*, const AdcSourceT AdcSource, const VoltageFormatT VoltageFormat);
    RFIDB1_StatusT (*SendMeasureDieTemperatureCommand)(RFIDB1_ObjectT*, const TemperatureFormatT TemperatureFormat);
    RFIDB1_StatusT (*SendSetIDACCurrentCommand)(RFIDB1_ObjectT*, const CurrentFormatT CurrentFormat, void *CurrentValue);
    RFIDB1_StatusT (*SendEnableComparatorCommand)(RFIDB1_ObjectT*, const ComparatorReferenceVoltageT InputConfig, const ComparatorOutputPinT OutputPinConfig, const ComparatorAsyncPacketEdgeSensitivityT PacketConfig, const uint8_t PowerSupplyDividerValue);
    RFIDB1_StatusT (*SendDisableComparatorCommand)(RFIDB1_ObjectT*);
    RFIDB1_StatusT (*SendEnablePWMCommand)(RFIDB1_ObjectT *object, PWMConfigFormatT PWMConfigFormat, void *Value, GpioT Gpio, uint8_t DutyCycle);
    RFIDB1_StatusT (*SendSetAESInitVectorCommand)(RFIDB1_ObjectT*, uint8_t *AESInitializationVector);
    RFIDB1_StatusT (*SendSetAESKeyCommand)(RFIDB1_ObjectT*, uint8_t *AESKey);
    RFIDB1_StatusT (*SendReadAESInitVectorCommand)(RFIDB1_ObjectT*);
    RFIDB1_StatusT (*SendReadAESKeyCommand)(RFIDB1_ObjectT*);
    RFIDB1_StatusT (*SendRawDataCommand)(RFIDB1_ObjectT*, uint8_t *Data, uint16_t Size);
    RFIDB1_StatusT (*SendWakeUpByte)(RFIDB1_ObjectT*);
    RFIDB1_StatusT (*SendStxByte)(RFIDB1_ObjectT*);
};

RFIDB1_StatusT GetRFIDB1Interface(RFIDB1_InterfaceT *interface);


#endif // RFIDB1-RFIDB1LIBRARY_H_INCLUDED
