/**
 *
 * @file      RFIDB1ClientProtocol.c
 * @brief     Protocol for RFIDB1 interface
 * @author    Marcin Baliniak, Piotr Szmelcer
 * @date      26/03/2019
 * @version   1.1
 * @copyright Eccel Technology Ltd
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <unistd.h>

#include "gpio.h"

#define GPIO_BASE_OLD            (0x20000000 + 0x200000) /* GPIO controller for old R-Pi version 1*/
#define GPIO_BASE_NEW            (0x3F000000 + 0x200000) /* GPIO controller for 2 and 3*/

#define PAGE_SIZE (4*1024)
#define BLOCK_SIZE (4*1024)

volatile unsigned *gpio = NULL;

// GPIO setup macros to acces GPIO registers. Always use INP_GPIO(x) before using OUT_GPIO(x) or SET_GPIO_ALT(x,y)
#define INP_GPIO(g) *(gpio+((g)/10)) &= ~(7<<(((g)%10)*3))
#define OUT_GPIO(g) *(gpio+((g)/10)) |=  (1<<(((g)%10)*3))
#define SET_GPIO_ALT(g,a) *(gpio+(((g)/10))) |= (((a)<=3?(a)+4:(a)==4?3:2)<<(((g)%10)*3))
#define GPIO_SET *(gpio+7)  // sets   bits which are 1 ignores bits which are 0
#define GPIO_CLR *(gpio+10) // clears bits which are 1 ignores bits which are 0
#define GET_GPIO(g) (*(gpio+13)&(1<<g)) // 0 if LOW, (1<<g) if HIGH

#define GPIO_PULL *(gpio+37) // Pull up/pull down
#define GPIO_PULLCLK0 *(gpio+38) // Pull up/pull down clock

/**
	@brief Function used to read GPIO state
	@param[in] num - number of GPIO
    @return status of line GPIO_LOW or GPIO_HIGH depend opn status
*/
int gpio_read_port(int num)
{
    if (!gpio)  //gpio_init not called before or failed
        return -1;

    if (GET_GPIO(num))
        return GPIO_HIGH;

    return GPIO_LOW;
}


/**
	@brief Function used to write GPIO state
	@param[in] num - number of GPIO
    @param[in] val - state of line, GPIO_LOW/GPIO_HIGH
    @return 0 on success and -1 in case of error
*/
int gpio_write_port(int num, int val)
{
    if (!gpio) //gpio_init not called before or failed
        return -1;

    if (val)
        GPIO_SET = 1<<num;
    else
        GPIO_CLR = 1<<num;

    return 0;
}

/**
	@brief Function used to setup GPIO port
	@param[in] num - number of GPIO
    @param[in] mode - GPIO_MODE_OUTPUT or GPIO_MODE_INPUT
    @return 0 on success and -1 in case of error
*/

int gpio_setup_port(int num, int mode)
{
    if (!gpio)   //gpio_init not called before or failed
        return -1;

    INP_GPIO(num); //must be called for out port also

    if (mode & GPIO_MODE_PULLUP)
        GPIO_PULL = 0x02; //enable pull up control
    else if (mode & GPIO_MODE_PULLDOWN)
        GPIO_PULL = 0x01; //enable pull down control
    else
        GPIO_PULL = 0x00; //disable pull up/down control

    usleep(100);
    GPIO_PULLCLK0 = 1 << num;
    usleep(100);
    GPIO_PULL = 0;  //disable pull up control
    GPIO_PULLCLK0 = 0;

    if (mode == GPIO_MODE_OUTPUT)
        OUT_GPIO(num);

    return 0;
}


/**

	@brief Function used to detect R-Pi version
    @return 0 for old version, and 1 for version 2 or 3, -1 - in case of error
*/

int get_rpi_version(void)
{
    char buff[4096], *p;
    FILE *fp;

    fp = fopen("/proc/cpuinfo", "r");
    if (fp == NULL)
        return -1;

    fread(buff, sizeof(buff), 1, fp);
    p = strstr(buff,"Revision");
    if (p)
    {
        p = strstr(p,":");
        if (p)
        {
            p+=2;
            if (strtol(p, NULL, 16) > 20)
                return 1;
        }
    }
    return 0;
}

/**

	@brief Function used to initialize GPIO subsystem, must be called before any
            other gpio functions
    @return 0 on success and -1 in case of error
*/

int gpio_init(void)
{
    int  mem_fd;
    void *gpio_map;
    int rpi_version;
    //open memory device
    if ((mem_fd = open("/dev/mem", O_RDWR|O_SYNC) ) < 0)
    {
        return -1;
    }

    rpi_version = get_rpi_version();

    //map GPIO address to acces GPIO memory
    if (rpi_version == 0)
        gpio_map = mmap(NULL, BLOCK_SIZE, PROT_READ|PROT_WRITE, MAP_SHARED, mem_fd, GPIO_BASE_OLD);
    else if (rpi_version == 1)
        gpio_map = mmap(NULL, BLOCK_SIZE, PROT_READ|PROT_WRITE, MAP_SHARED, mem_fd, GPIO_BASE_NEW);
    else
    {
        printf("Can't read Raspberry Pi version\n");
        return -1;
    }

    close(mem_fd); //No need to keep mem_fd open after mmap

    if (gpio_map == MAP_FAILED)
        return -1;

    // Always use volatile pointer!
    gpio = (volatile unsigned *)gpio_map;

    return 0;
}

