/**
 *
 * @file      main.c
 * @brief     Application to execute all RFID B1 commands from command line
 * @author    Marcin Baliniak
 * @date      26/03/2019
 * @copyright Eccel Technology Ltd
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include<string.h>
#include <unistd.h>
#include <termios.h>
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <time.h>
#include<sys/socket.h>
#include<arpa/inet.h>
#include <ctype.h>

#include "gpio.h"
#include "RFIDB1ClientInterface.h"

#define B1SerialBaudrate_Default 9600
#define RST_PIN     16
#define PWRDN_PIN   20
#define SLEEP_PIN   21

uint8_t ACK_frame[1024];
uint16_t ACK_frame_len;
bool ACK_received;

uint8_t ASYNC_byte;
bool ASYNC_received;

#define MAX_FRAME_SIZE 2048

int serial_fd = -1;
bool terminated = false;
bool test_started = false;

RFIDB1_InterfaceT B1Interface;
RFIDB1_ObjectT B1Object;

int std_output_fd = 1;

int own_printf(const char *format, ...)
{
    char buff[4096];
    int len;

    va_list args;
    va_start( args, format );
    len = vsnprintf(buff, sizeof(buff), format, args );

    if (std_output_fd == 1)
    {
        write(std_output_fd, buff, len);
    }
    else if(send(std_output_fd, buff, len, MSG_DONTWAIT)<0)
    {
        perror("\nSending failed.Error:");
    }
    va_end( args );
    return len;
}

/**
    @brief  Function used read nSleep pin state
    @param[in] object - Pointer to RFIDB1 object
    @return 0 - pin in low state
    @return 1 - pin in high state
    @details Function is needed to setup platform interface and menage modules state using PWRDN/nSleep/RST pins
*/
int getNSleep(RFIDB1_ObjectT *object)
{
    return gpio_read_port(SLEEP_PIN);
}

/**
    @brief  Function used to setup RST pin
    @param[in] object - Pointer to RFIDB1 object
    @param[in] val - new GPIO state
    @details Function is needed to setup platform interface and menage modules state using PWRDN/nSleep/RST pins
*/
void setResetPin(RFIDB1_ObjectT *object, int val)
{
    gpio_write_port(RST_PIN, val);
}

/**
    @brief  Function used to setup PWRDN pin
    @param[in] object - Pointer to RFIDB1 object
    @param[in] val - new GPIO state
    @details Function is needed to setup platform interface and menage modules state using PWRDN/nSleep/RST pins
*/
void setPowerPin(RFIDB1_ObjectT *object, int val)
{
    gpio_write_port(PWRDN_PIN, val);
}

/**
    @brief  Function  suspends  execution for requested milliseconds
    @param[in] object - Pointer to RFIDB1 object
    @param[in] val - milliseconds
    @details Function is needed to setup platform interface and menage modules state using PWRDN/nSleep/RST pins
*/
void delayMs(RFIDB1_ObjectT *object, int val)
{
    usleep(val * 1000);
}

/**
    @brief  Function used wake up module on RaspberryPI HAT
    @return 0 -  Success , -1 - Error
*/
int initializeGpioOnHat()
{
    //initialize GPIO subsystem
    if (gpio_init())
    {
        own_printf("Error - Can't initialize GPIO subsystem\n");
        return -1;
    }

    if (gpio_setup_port(RST_PIN, GPIO_MODE_OUTPUT))
    {
        own_printf("Error - Can't setup nRST pin\n");
        return -1;
    }

    if (gpio_setup_port(PWRDN_PIN, GPIO_MODE_OUTPUT))
    {
        own_printf("Error - Can't setup PWRDN pin\n");
        return -1;
    }

    if (gpio_setup_port(SLEEP_PIN, GPIO_MODE_INPUT | GPIO_MODE_PULLUP))
    {
        own_printf("Error - Can't setup nSleep pin\n");
        return -1;
    }

    return 0;
}


/**
    @brief Function used to open UART port
    @param[in] device - patch to devive
    @param[in] baudRate - selected baud rate
    @return device descriptor
    @return -1 if application can't open /dev/serial0 device
    @return -2 if incompatible baud is selected
*/
static int OpenPort(char *device, uint32_t baudRate)
{
    int uart_fd = -1;
    struct termios options;
    tcflag_t baud;

    uart_fd = open(device, O_RDWR | O_NOCTTY | O_NDELAY);		//Open in non blocking read/write mode,
    //serial0 is always link to UART on GPIO header
    if (uart_fd == -1)
    {
        own_printf("Error - Unable to open UART.  Ensure it is not in use by another application\n");
        return -1;
    }

    switch (baudRate)
    {
    case 1200:      baud = B1200; break;
    case 2400:      baud = B2400; break;
    case 4800:      baud = B4800; break;
    case 9600:      baud = B9600; break;
    case 19200:     baud = B19200;break;
    case 38400:     baud = B38400;break;
    case 57600:     baud = B57600;break;
    case 115200:    baud = B115200;break;
    case 230400:    baud = B230400;break;
    case 460800:    baud = B460800;break;
    case 921600:    baud = B921600;break;
    default:
        return -2;
    }

    tcgetattr(uart_fd, &options);
    options.c_cflag = baud | CS8 | CLOCAL | CREAD;        //<Set baud rate
    options.c_iflag = IGNPAR;
    options.c_oflag = 0;
    options.c_lflag = 0;
    tcflush(uart_fd, TCIFLUSH);
    tcsetattr(uart_fd, TCSANOW, &options);

    return uart_fd;
}

/**
    @brief Function used to send prepared data to UART hardware
    @param[in] object - Pointer to RFIDB1 object
    @param[in] data - data
    @param[in] size - data size
    @details Function is called after every UART command.
*/
void handleRequest(RFIDB1_ObjectT* object, uint8_t *data, uint16_t size)
{
    int *fd;

    fd = (int*)B1Interface.GetUserData(object);
    if (0)
    {
        int i;
        own_printf("writePacket %d bytes, fd %d\n", size, *fd);
        for (i = 0; i < size; i++)
            own_printf(" 0x%02X", data[i]);
        own_printf("\n");
    }


    if (write(*fd, data, size) <  size)
        own_printf("writePacket error writing!\n");
}

/**
    @brief Function used to pars incoming RFB1 data
    @param[in] object - Pointer to RFIDB1 object
    @param[in] data - data
    @param[in] size - data size
    @details In this function user can parse incoming data. Data does not contain any headers and is always unencrypted.
*/
void handleResponse(RFIDB1_ObjectT* object, uint8_t *data, uint16_t size)
{
    uint32_t i;

    if (0)
    {
        int i;
        own_printf("handleResponse %d bytes\n", size);
        for (i = 0; i < size; i++)
            own_printf(" 0x%02X", data[i]);
        own_printf("\n");
    }

    if (data[0] == B1ResponseACK)
    {
      ACK_frame_len = size-1;
      ACK_received = true;
      memcpy(ACK_frame, &data[1], size -1);

      if (0)
      {
        own_printf("<= ACK ");
        if (ACK_frame_len > 0)
        {
          for (i=0; i < ACK_frame_len; ++i)
            own_printf("0x%02X ", ACK_frame[i]);
        }
        own_printf("\n");
      }
    }
    else if (data[0] == B1ResponseAsyncPacket)
    {
      if (0)
        own_printf("<= ASYNC\n");
      ASYNC_byte = data[1];
      ASYNC_received = true;
    }

    fflush(stdout);
}

/**
    @brief  Function transform status from numeric to string
    @param[in] status - status in numeric format
    @return pointer to null terminated string
*/
char* getResultString(uint8_t status)
{
  switch (status)
  {
    case 0x00:  return "No ERROR";
    case 0x01:  return "Invalid command";
    case 0x02:  return "Invalid command parameter";
    case 0x03:  return "Indexes out of range";
    case 0x04:  return "Error when writing to non volatile memory";
    case 0x05:  return "System error";
    case 0x06:  return "Tag CRC error";
    case 0x07:  return "Tag collision";
    case 0x08:  return "Tag is not present";
    case 0x09:  return "Tag authentication error";
    case 0x0A:  return "Tag value block corrupted";
    case 0x0B:  return "Module overheated";
    case 0x0C:  return "Tag not supported";
    case 0x0D:  return "Tag communication error";
    case 0x0E:  return "Invalid password";
    case 0x0F:  return "Already locked";
    case 0xFF:  return "Module busy";
    default:  return "Unknown";
  }
}

/**
    @brief  Function waits for answer from module
    @param[in] needAck - function should wait for ACK frame
    @param[in] needAsync - function should wait for Async frame
    @param[in] timeout - timeout in ms
    @return true - requested frames received, false - timeout
*/
bool wait4Answer(bool needAck, bool needAsync, uint16_t timeout)
{
    fd_set rfds;
    struct timeval tv;
    int retval, fdmax, lenght;
    uint8_t buff[1024];

    ACK_frame_len = 0;
    ACK_received = false;
    ASYNC_received = false;

    while (1)
    {
        tv.tv_sec = timeout / 1000;
        tv.tv_usec = timeout%1000 * 1000;
        FD_ZERO(&rfds);
        FD_SET(serial_fd, &rfds);

        fdmax = serial_fd;

        retval = select(fdmax+1, &rfds, NULL, NULL, &tv);

        if (retval == -1)
        {
          perror("select()");
          own_printf("select(serial_fd: %d)\n", serial_fd);
          return false;
        }
        else if (retval)
        {
          if (FD_ISSET(serial_fd, &rfds))
          {
            do
            {
              lenght = read(serial_fd, buff, sizeof(buff));
              if (lenght > 0)
                B1Interface.ParseIncomingData(&B1Object, buff, lenght);
            } while (lenght > 0);


            if (ASYNC_received == needAsync && ACK_received == needAck)
              return true;
          }
        }
        else
        {
            return false;
        }
    }

}

/**
    @brief  Function reads UID from tag
    @param[out] uid - output buffer pointer to store UID
    @param[out] len - pointer to store UID length
    @return Tag type
*/
uint8_t getUid(uint8_t *uid, uint8_t *len)
{
    uint8_t command, i, tagType = TagType_NoTag;

    command = B1RFIDCommandGetUIDAndType;
    B1Interface.SendWriteToRFIDMemoryCommand(&B1Object, B1RegisterCommand, &command, 1);

    if (wait4Answer(true, true, 200) && (ASYNC_byte & B1AsyncPacketParamBit_RFIDCommand))
    {
      B1Interface.SendReadFromRFIDMemoryCommand(&B1Object, B1RegisterTagUID, 12);
      if (!wait4Answer(true, false, 200))
        own_printf("ReadFromRFIDMemory failed: no ACK\n");

      tagType = ACK_frame[10];
      *len = ACK_frame[11];
      memcpy(uid, ACK_frame, *len);
    }

    return tagType;
}

/**
    @brief  Function used to turn off RFID antenna and stop RF communication
    @return 0 - Success, -1 - Error
*/
int halt(void)
{
  uint8_t command;

  command = B1RFIDCommandHalt;

  B1Interface.SendWriteToRFIDMemoryCommand(&B1Object, B1RegisterCommand, &command, 1);

  if (wait4Answer(true, true, 200))
  {
    if (!ACK_received)
      own_printf("No ACK for HALT command");
    if (!ACK_received)
      own_printf("No ASYNC for HALT command");

    return -1;
  }
  return 0;
}

/**
    @brief  Function prints tag
    @param[in] tagType - tagtype
    @param[in] uid - pointer to buffer with UID
    @param[in] len - UID length
*/
void print_tag(uint8_t tagType, uint8_t *uid, uint8_t len)
{
    int i;
    own_printf("Tag type: ");
    switch (tagType)
    {
    case TagType_NoTag:             own_printf("NoTag");            break;
    case TagType_NotComplete:       own_printf("NotComplete");      break;
    case TagType_Ultralight:        own_printf("Ultralight");       break;
    case TagType_UltralightEV1_80B: own_printf("UltralightEV1_80B");break;
    case TagType_UltralightEV1_164B:own_printf("UltralightEV1_164B");break;
    case TagType_ClassicMini:       own_printf("ClassicMini");      break;
    case TagType_Classic1K:         own_printf("Classic1K");        break;
    case TagType_Classic4K:         own_printf("Classic4K");        break;
    case TagType_NTAG203F:          own_printf("NTAG203F");         break;
    case TagType_NTAG210:           own_printf("NTAG210");          break;
    case TagType_NTAG212:           own_printf("NTAG212");          break;
    case TagType_NTAG213F:          own_printf("NTAG213F");         break;
    case TagType_NTAG216F:          own_printf("NTAG216F");         break;
    case TagType_NTAG213:           own_printf("NTAG213");          break;
    case TagType_NTAG215:           own_printf("NTAG215");          break;
    case TagType_NTAG216:           own_printf("NTAG216");          break;
    default:
        own_printf("Unknown");
        break;
    }

    if (tagType != TagType_NoTag)
    {
        own_printf("\nUID size: %d\n", len);
        own_printf("UID: ");
        for (i=0; i < len; ++i)
          own_printf("%02X", uid[i]);
    }

    own_printf("\n");
}

/**
    @brief Function unlocks module memory
    @param[in] password - buffer with password in ASCII-HEX format
    @return 0 - Success, -1 - Error
*/
int unlockModule(char *password)
{
  uint8_t command[16];
  unsigned int h[8];
  int i;

  if (strlen(password) != 16)
  {
    own_printf( "\nPassword must have 16 chars!\n");
    return -1;
  }

  for (i = 0; i < 8; i++)
  {
    if (sscanf(&password[i*2], "%02x", &h[i]) != 1)
    {
      own_printf( "\nOnly HEX chars are allowed from [0-F]!\n");
      return -1;
    }
  }

  command[0] = B1RFIDCommandUnlock;
  own_printf("Unlocking module with password:");
  for (i = 0; i < 8; i++)
  {
    own_printf("%02X", h[i]);
    command[i+1] = h[i];
  }

  own_printf("\n");

  B1Interface.SendWriteToRFIDMemoryCommand(&B1Object, B1RegisterCommand, command, 9);

  if (!wait4Answer(true, true, 200))
  {
    own_printf("SendWriteToRFIDMemory failed: no ACK\n");
    return -1;
  }


  B1Interface.SendReadFromRFIDMemoryCommand(&B1Object, B1RegisterResult, 1);
  if (!wait4Answer(true, false, 200))
  {
    own_printf("ReadFromRFIDMemory failed: no ACK\n");
    return -1;
  }

  if (ACK_frame[0] == 0x00)
  {
    own_printf("Module unlocked.\n");
    return 0;
  }
  else
  {
    own_printf("%s\n", getResultString(ACK_frame[0]));
  }

  return -1;
}

/**
    @brief Function locks module memory
    @return 0 - Success, -1 - Error
*/
int lockModule()
{
  uint8_t command[16];
  unsigned int h[8];
  int i;

  command[0] = B1RFIDCommandLock;

  B1Interface.SendWriteToRFIDMemoryCommand(&B1Object, B1RegisterCommand, command, 1);

  if (!wait4Answer(true, true, 200))
  {
    own_printf("SendWriteToRFIDMemory failed: no ACK\n");
    return -1;
  }


  B1Interface.SendReadFromRFIDMemoryCommand(&B1Object, B1RegisterResult, 1);
  if (!wait4Answer(true, false, 200))
  {
    own_printf("ReadFromRFIDMemory failed: no ACK\n");
    return -1;
  }

  if (ACK_frame[0] == 0x00)
  {
    own_printf("Module locked.\n");
    return 0;
  }
  else
  {
    own_printf("%s\n", getResultString(ACK_frame[0]));
  }

  return -1;
}

/**
    @brief Function resets module to factory defaults
    @param[in] mode - reset mode, "1" - factory reset using GPIO pins on Raspberry PI HAT
    @return 0 - Success, -1 - Error
*/
int reset2Defaults(char *mode)
{
  uint8_t command[16];
  unsigned int h[8];
  int i;

  if (mode && mode[0] == '1')
  {
    if (initializeGpioOnHat())
        return -1;

    if (B1Interface.ResetToDefaults(&B1Object) == B1StatusT_OK)
    {
        own_printf("RFIDB1 module on Raspberry Pi HAT restored to defaults.\n");
        return 0;
    }
    else
    {
        own_printf("Can't reset RFIDB1 module to defaults (works only on Raspberry Pi HAT!).\n");
        return -1;
    }
  }

  command[0] = B1RFIDCommandResetToFactoryDefaults;

  B1Interface.SendWriteToRFIDMemoryCommand(&B1Object, B1RegisterCommand, command, 1);

  if (!wait4Answer(true, true, 200))
  {
    own_printf("SendWriteToRFIDMemory failed: no ACK\n");
    return -1;
  }


  B1Interface.SendReadFromRFIDMemoryCommand(&B1Object, B1RegisterResult, 1);
  if (!wait4Answer(true, false, 200))
  {
    own_printf("ReadFromRFIDMemory failed: no ACK\n");
    return -1;
  }

  if (ACK_frame[0] == 0x00)
  {
    own_printf("Module locked.\n");
    return 0;
  }
  else
  {
    own_printf("%s\n", getResultString(ACK_frame[0]));
  }

  return -1;
}


/**
    @brief Function reads from module memory
    @param[in] address - memory address
    @param[in] len - length to read
    @return 0 - Success, -1 - Error
*/
int readFromRfidMemory(uint16_t address, uint16_t len)
{
  B1Interface.SendReadFromRFIDMemoryCommand(&B1Object, address, len);
  if (!wait4Answer(true, false, 200))
  {
    own_printf("ReadFromRFIDMemory failed: no ACK\n");
    return -1;
  }

  return 0;
}

/**
    @brief Function writes to module memory
    @param[in] address - memory address
    @param[in] buff - pointer with data to write
    @param[in] len - number of bytes to write
    @return 0 - Success, -1 - Error
*/
int writeToRfidMemory(uint16_t address, uint8_t *buff, uint16_t len)
{
  B1Interface.SendWriteToRFIDMemoryCommand(&B1Object, address, buff, len);
  if (!wait4Answer(true, false, 200))
  {
    if (!ACK_received)
      own_printf("Write to B1RegisterAuthKeyPass failed: no ACK\n");

    return -1;
  }

  B1Interface.SendReadFromRFIDMemoryCommand(&B1Object, address, len);
  if (!wait4Answer(true, false, 200))
  {
    own_printf("ReadFromRFIDMemory failed: no ACK\n");
    return -1;
  }

  if (memcmp(buff, ACK_frame, len))
  {
    own_printf("Verification failed!\n");
    return -1;
  }

  own_printf("Verification passed - write OK.\n");

  return 0;
}

/**
    @brief Function prints module version
    @return 0 - Success, -1 - Error
*/
int getModuleVersion()
{
  int keyNum = -1, i;
  uint8_t cmd[30];

  cmd[0] = B1RFIDCommandGetModuleVersion;

  B1Interface.SendWriteToRFIDMemoryCommand(&B1Object, B1RegisterCommand, cmd, 1);

  if (wait4Answer(true, true, 200) && (ASYNC_byte & B1AsyncPacketParamBit_RFIDCommand))
  {
    //read the status
    B1Interface.SendReadFromRFIDMemoryCommand(&B1Object, B1RegisterResult, 1);
    if (!wait4Answer(true, false, 200))
    {
      own_printf("Get module version failed: no ACK\n");
      return -1;
    }

    if (ACK_frame[0] != 0x00)
    {
      own_printf("Get module version authentication error: %s\n", getResultString(ACK_frame[0]));
      return -1;
    }
  }
  else
  {
    if (!ACK_received)
      own_printf("No ACK for read command\n");

    if (!ASYNC_received)
      own_printf("No ASYNC for read block command\n");

    return -1;
  }

  if (readFromRfidMemory(32, 128))
    return -1;

  own_printf("Version is: %s\n", (char*)ACK_frame);

  return 0;
}

/**
    @brief Function used to change module password
    @param[in] password - new password in ASCII-HEX format
    @return 0 - Success, -1 - Error
*/
int changePassword(char *password)
{
  int i;
  unsigned int p_uint[8];
  uint8_t p_uint8[8];

  if (strlen(password) != 16)
  {
    own_printf( "\nKey must have 16 chars!\n");
    return -1;
  }

  for (i = 0; i < 8; i++)
  {
    if (sscanf(&password[i*2], "%02x", &p_uint[i]) != 1)
    {
      own_printf( "\nOnly HEX chars are allowed from [0-F]!\n");
      return -1;
    }
  }

  own_printf("Password is:");
  for (i = 0; i < 8; i++)
  {
    p_uint8[i] = p_uint[i] & 0xff;
    own_printf("%02X", p_uint8[i]);
  }
  own_printf("\n");

  return writeToRfidMemory(B1RegisterPassword, p_uint8, 8);
}

/**
    @brief Function reads key from module memory
    @param[in] number - key number
    @return 0 - Success, -1 - Error
*/
int readKey(char *number)
{
  int num = atoi(number), k;
  uint8_t key[8];
  if (num < 0 || num > 39)
  {
    own_printf("Key number must be in range [0-39]\n");
    return -1;
  }

  if (readFromRfidMemory(B1RegisterAuthKeyPass00 + num*6, 6))
    return -1;

  memcpy(key, ACK_frame, 6);

  own_printf("Key %d is: ", num);
  for (k=0; k< 6; k++)
    own_printf("%02X", key[k]);
  own_printf("\n");
  return 0;
}

/**
    @brief Function writes key to module memory
    @param[in] number - key number
    @param[in] key - key in ASCII-HEX format
    @return 0 - Success, -1 - Error
*/
int writeKey(char *number, char *key)
{
  int num = atoi(number), i;
  unsigned int k_uint[6];
  uint8_t k_uint8[6];

  if (num < 0 || num > 39)
  {
    own_printf("Key number must be in range [0-39]\n");
    return -1;
  }

  if (strlen(key) != 12)
  {
    own_printf( "\nKey must have 12 chars!\n");
    return -1;
  }

  for (i = 0; i < 6; i++)
  {
    if (sscanf(&key[i*2], "%02x", &k_uint[i]) != 1)
    {
      own_printf( "\nOnly HEX chars are allowed from [0-F]!\n");
      return -1;
    }
  }

  own_printf("Key (%d) :", num);
  for (i = 0; i < 6; i++)
  {
    k_uint8[i] = k_uint[i];
    own_printf("%02X", k_uint8[i]);
  }
  own_printf("\n");

  return writeToRfidMemory(B1RegisterAuthKeyPass00 + num*6, k_uint8, 6);
}

/**
    @brief Function reads AES key from module memory
    @param[in] number - key number
    @return 0 - Success, -1 - Error
*/
int readAesKey(char *number)
{
  int num = atoi(number), k;
  uint8_t key[16];
  if (num < 0 || num > 1)
  {
    own_printf("AES key number must be in range [0-1]\n");
    return -1;
  }

  if (readFromRfidMemory(B1RegisterAESKey0 + num*16, 16))
    return -1;

  memcpy(key, ACK_frame, 16);

  own_printf("Key %d is: ", num);
  for (k=0; k< 16; k++)
    own_printf("%02X", key[k]);
  own_printf("\n");
  return 0;
}

/**
    @brief Function used to write AES key to module memory
    @param[in] number - key number
    @param[in] key - key in ASCII-HEX format
    @return 0 - Success, -1 - Error
*/
int writeAesKey(char *number, char *key)
{
  int num = atoi(number), i;
  unsigned int k_uint[16];
  uint8_t k_uint8[16];

  if (num < 0 || num > 1)
  {
    own_printf("Key number must be in range [0-1]\n");
    return -1;
  }

  if (strlen(key) != 32)
  {
    own_printf( "\nKey must have 32 chars (16-hex)!\n");
    return -1;
  }

  for (i = 0; i < 16; i++)
  {
    if (sscanf(&key[i*2], "%02x", &k_uint[i]) != 1)
    {
      own_printf( "\nOnly HEX chars are allowed from [0-F]!\n");
      return -1;
    }
  }

  own_printf("Key (%d) :", num);
  for (i = 0; i < 16; i++)
  {
    k_uint8[i] = k_uint[i];
    own_printf("%02X", k_uint8[i]);
  }
  own_printf("\n");

  return writeToRfidMemory(B1RegisterAESKey0 + num*16, k_uint8, 16);
}

/**
    @brief Function used to read AES vector to module memory
    @param[in] number - vector number
    @return 0 - Success, -1 - Error
*/
int readAesVector(char *number)
{
  int num = atoi(number), k;
  uint8_t vector[16];
  if (num < 0 || num > 1)
  {
    own_printf("AES vector number must be in range [0-1]\n");
    return -1;
  }

  if (readFromRfidMemory(B1RegisterAESInitVector0 + num*16, 16))
    return -1;

  memcpy(vector, ACK_frame, 16);

  own_printf("Vector %d is: ", num);
  for (k=0; k< 16; k++)
    own_printf("%02X", vector[k]);
  own_printf("\n");
  return 0;
}

/**
    @brief Function used to write AES vector to module memory
    @param[in] number - key number
    @param[in] vector - vector in ASCII-HEX format
    @return 0 - Success, -1 - Error
*/
int writeAesVector(char *number, char *vector)
{
  int num = atoi(number), i;
  unsigned int k_uint[16];
  uint8_t k_uint8[16];

  if (num < 0 || num > 1)
  {
    own_printf("Vector number must be in range [0-1]\n");
    return -1;
  }

  if (strlen(vector) != 32)
  {
    own_printf( "\nKey must have 32 chars (16-hex)!\n");
    return -1;
  }

  for (i = 0; i < 16; i++)
  {
    if (sscanf(&vector[i*2], "%02x", &k_uint[i]) != 1)
    {
      own_printf( "\nOnly HEX chars are allowed from [0-F]!\n");
      return -1;
    }
  }

  own_printf("Vector (%d) :", num);
  for (i = 0; i < 16; i++)
  {
    k_uint8[i] = k_uint[i];
    own_printf("%02X", k_uint8[i]);
  }
  own_printf("\n");

  return writeToRfidMemory(B1RegisterAESInitVector0 + num*16, k_uint8, 16);
}

/**
    @brief Function used to read data buffer from module memory
    @param[in] addr - address to read
    @param[in] len - length to read
    @return 0 - Success, -1 - Error
*/
int readData(char *addr, char *len)
{
  int address = atoi(addr), k;
  int length = atoi(len);

  if (address < 0 || address > 256)
  {
    own_printf("Address must be in range [0-256].\n");
    return -1;
  }

  if (address + length > 256)
  {
    own_printf("You trying to read out of data buffer.\n");
    return -1;
  }

  if (readFromRfidMemory(address + 0x20, length))
    return -1;

  own_printf("Reading %d bytes from data buffer from address %d:", length, address);
  for (k=0; k< length; k++)
    own_printf("%02X", ACK_frame[k]);
  own_printf("\n");

  return 0;
}

/**
    @brief Function used to write data to module memory
    @param[in] addr - address to read
    @param[in] hex - ASCI-HEX string to write
    @return 0 - Success, -1 - Error
*/
int writeData(char *addr, char *hex)
{
  int address = atoi(addr), i;
  int length = strlen(hex);
  unsigned int k_uint[728];
  uint8_t k_uint8[728];

  if (address < 0 || address > 728)
  {
    own_printf("Address must be in range [0-728].\n");
    return -1;
  }

  if (address + length/2 > 728)
  {
    own_printf("You trying to write out of module memory.\n");
    return -1;
  }

  if (length%2)
  {
    own_printf("Hex string length must be even.");
    return -1;
  }

  for (i = 0; i < length/2; i++)
  {
    if (sscanf(&hex[i*2], "%02x", &k_uint[i]) != 1)
    {
      own_printf( "\nOnly HEX chars are allowed from [0-F]!\n");
      return -1;
    }
  }

  for (i = 0; i < length/2; i++)
    k_uint8[i] = k_uint[i];

  own_printf("Writing %d bytes to data buffer at address %d\n", address, length/2);

  return writeToRfidMemory(address + 0x20, k_uint8, length/2);
}

/**
    @brief Function used to read module memory
    @param[in] addr - address to read
    @param[in] len - length to read
    @return 0 - Success, -1 - Error
*/
int rawRead(char *addr, char *len)
{
  int address = atoi(addr), k;
  int length = atoi(len);

  if (address < 0 || address > 728)
  {
    own_printf("Address must be in range [0-728].\n");
    return -1;
  }

  if (address + length > 728)
  {
    own_printf("You trying to read out of module memory.\n");
    return -1;
  }

  if (readFromRfidMemory(address, length))
    return -1;

  own_printf("Reading %d bytes from address %d:", length, address);
  for (k=0; k< length; k++)
    own_printf("%02X", ACK_frame[k]);
  own_printf("\n");

  return 0;
}

/**
    @brief Function used to write module memory
    @param[in] addr - address to read
    @param[in] hex - ASCI-HEX string to write
    @return 0 - Success, -1 - Error
*/
int rawWrite(char *addr, char *hex)
{
  int address = atoi(addr), i;
  int length = strlen(hex);
  unsigned int k_uint[728];
  uint8_t k_uint8[728];

  if (address < 0 || address > 728)
  {
    own_printf("Address must be in range [0-728].\n");
    return -1;
  }

  if (address + length/2 > 728)
  {
    own_printf("You trying to write out of module memory.\n");
    return -1;
  }

  if (length%2)
  {
    own_printf("Hex string length must be even.");
    return -1;
  }

  for (i = 0; i < length/2; i++)
  {
    if (sscanf(&hex[i*2], "%02x", &k_uint[i]) != 1)
    {
      own_printf( "\nOnly HEX chars are allowed from [0-F]!\n");
      return -1;
    }
  }

  for (i = 0; i < length/2; i++)
    k_uint8[i] = k_uint[i];

  own_printf("Writing %d bytes to address %d\n", address, length/2);

  return writeToRfidMemory(address, k_uint8, length/2);
}

/**
    @brief Function used to copy module memory
    @param[in] cdest - string with destination address
    @param[in] csrc - string with destination address
    @param[in] hex - ASCI-HEX string to write
    @return 0 - Success, -1 - Error
*/
int copyData(char *cdest, char *csrc, char *clen)
{
  int k;
  uint16_t src = (uint16_t)atoi(csrc);
  uint16_t dest = (uint16_t)atoi(cdest);
  uint8_t len = (uint8_t)atoi(clen);
  uint8_t cmd[30];

  if (src < 0 || src > 728)
  {
    own_printf("Address need to be in range 0-728\n");
    return -1;
  }

  if (dest + len > 728)
  {
    own_printf("You trying to calculate copy out of memory\n");
    return -1;
  }

  cmd[0] = B1RFIDCommandCopyData;

  cmd[1] = dest & 0xff;
  cmd[2] = (dest >> 8) & 0xff;

  cmd[3] = src & 0xff;
  cmd[4] = (src >> 8) & 0xff;

  cmd[5] = len & 0xff;
  cmd[6] = (len >> 8) & 0xff;

  B1Interface.SendWriteToRFIDMemoryCommand(&B1Object, B1RegisterCommand, cmd, 7);

  if (wait4Answer(true, true, 200) && (ASYNC_byte & B1AsyncPacketParamBit_RFIDCommand))
  {
    //read the status
    B1Interface.SendReadFromRFIDMemoryCommand(&B1Object, B1RegisterResult, 1);
    if (!wait4Answer(true, false, 200))
    {
      own_printf("Copy data failed: no ACK\n");
      return -1;
    }

    if (ACK_frame[0] != 0x00)
    {
      own_printf("Write error: %s\n", getResultString(ACK_frame[0]));
      return -1;
    }
  }
  else
  {
    if (!ACK_received)
      own_printf("No ACK for write command\n");

    if (!ASYNC_received)
      own_printf("No ASYNC for write block command\n");

    return -1;
  }

  return 0;
}


/**
    @brief Function used to crypt/decrypt module memory
    @param[in] ckey - ASCI-HEX string with crypt key
    @param[in] cvector - ASCI-HEX string with crypt vector
    @param[in] coffset - memory offset
    @param[in] cblocks - number of blocks
    @param[in] decrypt - bool flag to select crypt/decrypt procedure
    @return 0 - Success, -1 - Error
*/
int cryptData(char *ckey, char *cvector, char *coffset, char *cblocks, bool decrypt)
{
  int k;
  int key = atoi(ckey);
  int vector = atoi(cvector);
  int offset = atoi(coffset);
  int blocks = atoi(cblocks);
  uint8_t cmd[5];

  if (key < 0 || key > 1)
  {
    own_printf("Key number need to be 0 or 1\n");
    return -1;
  }

  if (vector < 0 || vector > 1)
  {
    own_printf("Key number need to be 0 or 1\n");
    return -1;
  }

  if (blocks < 1 || blocks > 8)
  {
    own_printf("Number of blocks need to be in range 1-8. (every block have 16-bytes)\n");
    return -1;
  }

  if (offset + blocks * 16 > 728)
  {
    own_printf("You trying to encrypt data out of module memory.\n");
    return -1;
  }

  if (decrypt)
    cmd[0] = B1RFIDCommandDecryptData;
  else
    cmd[0] = B1RFIDCommandEncryptData;

  cmd[1] = key;
  cmd[2] = vector;
  cmd[3] = offset;
  cmd[4] = blocks;

  B1Interface.SendWriteToRFIDMemoryCommand(&B1Object, B1RegisterCommand, cmd, 5);

  if (wait4Answer(true, true, 200) && (ASYNC_byte & B1AsyncPacketParamBit_RFIDCommand))
  {
    //read the status
    B1Interface.SendReadFromRFIDMemoryCommand(&B1Object, B1RegisterResult, 1);
    if (!wait4Answer(true, false, 200))
    {
      own_printf("B1RFIDCommandEncryptData failed: no ACK\n");
      return -1;
    }

    if (ACK_frame[0] != 0x00)
    {
      own_printf("Write error: %s\n", getResultString(ACK_frame[0]));
      return -1;
    }
  }
  else
  {
    if (!ACK_received)
      own_printf("No ACK for write command\n");

    if (!ASYNC_received)
      own_printf("No ASYNC for write block command\n");

    return -1;
  }

  return 0;
}

/**
    @brief Function used to calculate CRC from data stored in module memory
    @param[in] caddr - start address
    @param[in] clength - data length
    @param[in] coffset - offset to store CRC data
    @return 0 - Success, -1 - Error
*/
int crcData(char *caddr, char *clength, char *coffset)
{
  int k;
  int addr = atoi(caddr);
  int length = atoi(clength);
  int offset = atoi(coffset);
  uint8_t cmd[6];

  if (addr < 0 || addr > 728)
  {
    own_printf("Address need to be in range 0-728\n");
    return -1;
  }

  if (addr + length > 728)
  {
    own_printf("You trying to calculate CRC out of memory\n");
    return -1;
  }

  if (offset < 0 || offset > 256)
  {
    own_printf("Offset should be in range 0-256\n");
    return -1;
  }

  cmd[0] = B1RFIDCommandCalculateCRC;

  cmd[1] = addr & 0xff;
  cmd[2] = (addr >> 8) & 0xff;

  cmd[3] = length & 0xff;
  cmd[4] = (length >> 8) & 0xff;

  cmd[5] = offset;

  B1Interface.SendWriteToRFIDMemoryCommand(&B1Object, B1RegisterCommand, cmd, 6);

  if (wait4Answer(true, true, 200) && (ASYNC_byte & B1AsyncPacketParamBit_RFIDCommand))
  {
    //read the status
    B1Interface.SendReadFromRFIDMemoryCommand(&B1Object, B1RegisterResult, 1);
    if (!wait4Answer(true, false, 200))
    {
      own_printf("B1RFIDCommandCalculateCRC failed: no ACK\n");
      return -1;
    }

    if (ACK_frame[0] != 0x00)
    {
      own_printf("Write error: %s\n", getResultString(ACK_frame[0]));
      return -1;
    }
  }
  else
  {
    if (!ACK_received)
      own_printf("No ACK for write command\n");

    if (!ASYNC_received)
      own_printf("No ASYNC for write block command\n");

    return -1;
  }

  if (readFromRfidMemory(B1RegisterDataBuffer + offset, 2))
    return -1;

  own_printf("CRC stored in data buffer with offset %d bytes is:", offset);
  for (k=0; k< 2; k++)
    own_printf("%02X", ACK_frame[k]);
  own_printf("\n");

  return 0;
}

//------------------------------ RFID functions --------------------------------------------
/**
    @brief Function used to read blocks from Mifare tag
    @param[in] cblock_num - start block
    @param[in] cblocks2read - number of blocks to read
    @param[in] cdataAddr - address to store read data
    @param[in] ckeyAB - type of key A/B
    @param[in] ckey - key used to read memory
    @param[in] dataOnly - skip special blocks
    @return 0 - Success, -1 - Error
*/
int readBlock(char *cblock_num, char *cblocks2read,char *cdataAddr, char *ckeyAB, char *ckey, bool dataOnly)
{
  int blockNum = atoi(cblock_num);
  int blocks2read = atoi(cblocks2read);
  int dataAddr = atoi(cdataAddr);
  char keyAB = *ckeyAB;
  unsigned int key_uint;
  uint8_t key_uint8[6];
  int keyNum = -1, i;
  uint8_t cmd[30];

  if (blocks2read * 16 + dataAddr > 256)
  {
    own_printf("You trying to read out of data buffer\n");
    return -1;
  }

  if (keyAB != 'A' && keyAB != 'B')
  {
    own_printf("Please provide correct key A or B\n");
    return -1;
  }

  if (strlen(ckey) <= 2)
  {
    //using build in key
    keyNum = atoi(ckey);
    if (keyNum < 0 || keyNum > 39)
    {
      own_printf( "\nKey numer must be in range [0-39]!\n");
      return -1;
    }
  }
  else if (strlen(ckey) == 12)
  {
    for (i = 0; i < 6; i++)
    {
      if (sscanf(&ckey[i*2], "%02x", &key_uint) != 1)
      {
        own_printf( "\nOnly HEX chars are allowed from [0-F]!\n");
        return -1;
      }
      else
        key_uint8[i] = key_uint;
    }
  }
  else
  {
    own_printf("You should enter key as key number from module memory [range 0-39], or 12 digit hex string\n");
    return -1;
  }

  own_printf("Reading %d blocks, starting from %d. Data stored in data buffer address %d. ", blockNum, blocks2read, dataAddr);

  if (dataOnly)
    cmd[0] = B1RFIDCommandReadDataBlock;
  else
    cmd[0] = B1RFIDCommandReadBlock;

  cmd[1] = blockNum;
  cmd[2] = blocks2read;
  cmd[3] = dataAddr;

  if (keyNum != -1)
  {
    own_printf("Using key %c (key number %s stored in module memory)\n", keyAB, ckey);
    cmd[4] = ((uint8_t) keyNum);
  }
  else
  {
    own_printf("Using key %c (key %s)\n", keyAB, ckey);
    cmd[4] = 1 << 6;
    memcpy(&cmd[5], key_uint8, 6);
  }

  if (keyAB == 'B')
    cmd[4] |= 1 << 7;

  B1Interface.SendWriteToRFIDMemoryCommand(&B1Object, B1RegisterCommand, cmd, 11);

  if (wait4Answer(true, true, 200) && (ASYNC_byte & B1AsyncPacketParamBit_RFIDCommand))
  {
    //read the status
    B1Interface.SendReadFromRFIDMemoryCommand(&B1Object, B1RegisterResult, 1);
    if (!wait4Answer(true, false, 200))
    {
      own_printf("ReadFromRFIDMemory failed: no ACK\n");
      return -1;
    }

    if (ACK_frame[0] != 0x00)
    {
      own_printf("Read error: %s\n", getResultString(ACK_frame[0]));
      return -1;
    }
  }
  else
  {
    if (!ACK_received)
      own_printf("No ACK for read command\n");

    if (!ASYNC_received)
      own_printf("No ASYNC for read block command\n");

    return -1;
  }

  if (readFromRfidMemory(dataAddr + 32, blocks2read * 16))
    return -1;


  own_printf("Data bytes:");
  for (i = 0; i < blocks2read*16; i++)
    own_printf("%02X", ACK_frame[i]);
  own_printf("\n");

  return 0;
}

/**
    @brief Function used to write blocks to Mifare tag
    @param[in] cblock_num - start block
    @param[in] cblocks2read - number of blocks to read
    @param[in] cdataAddr - address to store read data
    @param[in] ckeyAB - type of key A/B
    @param[in] ckey - key used to read memory
    @param[in] dataOnly - skip special blocks
    @return 0 - Success, -1 - Error
*/
int writeBlock(char *cblock_num, char *cblocks2write,char *cdataAddr, char *ckeyAB, char *ckey, bool dataOnly)
{
  int blockNum = atoi(cblock_num);
  int blocks2write = atoi(cblocks2write);
  int dataAddr = atoi(cdataAddr);
  char keyAB = *ckeyAB;
  unsigned int key_uint;
  uint8_t key_uint8[6];
  int keyNum = -1, i;
  uint8_t cmd[30];

  if (blocks2write * 16 + dataAddr > 256)
  {
    own_printf("You trying to write out of data buffer\n");
    return -1;
  }

  if (keyAB != 'A' && keyAB != 'B')
  {
    own_printf("Please provide correct key A or B\n");
    return -1;
  }

  if (strlen(ckey) <= 2)
  {
    //using build in key
    keyNum = atoi(ckey);
    if (keyNum < 0 || keyNum > 39)
    {
      own_printf( "\nKey numer must be in range [0-39]!\n");
      return -1;
    }
  }
  else if (strlen(ckey) == 12)
  {
    for (i = 0; i < 6; i++)
    {
      if (sscanf(&ckey[i*2], "%02x", &key_uint) != 1)
      {
        own_printf( "\nOnly HEX chars are allowed from [0-F]!\n");
        return -1;
      }
      else
        key_uint8[i] = key_uint;
    }
  }
  else
  {
    own_printf("You should enter key as key number from module memory [range 0-39], or 12 digit hex string\n");
    return -1;
  }

  own_printf("Writing %d blocks, starting from %d. Writing data from data buffer address %d. ", blockNum, blocks2write, dataAddr);

  if (dataOnly)
    cmd[0] = B1RFIDCommandWriteDataBlock;
  else
    cmd[0] = B1RFIDCommandWriteBlock;

  cmd[1] = blockNum;
  cmd[2] = blocks2write;
  cmd[3] = dataAddr;

  if (keyNum != -1)
  {
    own_printf("Using key %c (key number %s stored in module memory)\n", keyAB, ckey);
    cmd[4] = ((uint8_t) keyNum);
  }
  else
  {
    own_printf("Using key %c (key %s)\n", keyAB, ckey);
    cmd[4] = 1 << 6;
    memcpy(&cmd[5], key_uint8, 6);
  }

  if (keyAB == 'B')
    cmd[4] |= 1 << 7;

  B1Interface.SendWriteToRFIDMemoryCommand(&B1Object, B1RegisterCommand, cmd, 11);

  if (wait4Answer(true, true, 200) && (ASYNC_byte & B1AsyncPacketParamBit_RFIDCommand))
  {
    //read the status
    B1Interface.SendReadFromRFIDMemoryCommand(&B1Object, B1RegisterResult, 1);
    if (!wait4Answer(true, false, 200))
    {
      own_printf("B1RFIDCommandWriteBlock failed: no ACK\n");
      return -1;
    }

    if (ACK_frame[0] != 0x00)
    {
      own_printf("Write error: %s\n", getResultString(ACK_frame[0]));
      return -1;
    }
  }
  else
  {
    if (!ACK_received)
      own_printf("No ACK for write command\n");

    if (!ASYNC_received)
      own_printf("No ASYNC for write block command\n");

    return -1;
  }

  return 0;
}

/**
    @brief Function used to read pages from tag
    @param[in] cblock_num - start block
    @param[in] cblocks2read - number of blocks to read
    @param[in] cdataAddr - address to store read data
    @return 0 - Success, -1 - Error
*/
int readPage(char *cblock_num, char *cblocks2read,char *cdataAddr)
{
  int blockNum = atoi(cblock_num);
  int blocks2read = atoi(cblocks2read);
  int dataAddr = atoi(cdataAddr);
  int i;
  uint8_t cmd[30];

  if (blockNum + blocks2read > 230)
  {
    own_printf("You trying to read out tag memory\n");
    return -1;
  }

  if (blocks2read * 4 + dataAddr > 256)
  {
    own_printf("You trying to read out of data buffer\n");
    return -1;
  }

  own_printf("Reading %d pages, starting from %d. Data stored in data buffer address %d.\n", blockNum, blocks2read, dataAddr);

  cmd[0] = B1RFIDCommandReadPage;
  cmd[1] = blockNum;
  cmd[2] = blocks2read;
  cmd[3] = dataAddr;


  B1Interface.SendWriteToRFIDMemoryCommand(&B1Object, B1RegisterCommand, cmd, 4);

  if (wait4Answer(true, true, 200) && (ASYNC_byte & B1AsyncPacketParamBit_RFIDCommand))
  {
    //read the status
    B1Interface.SendReadFromRFIDMemoryCommand(&B1Object, B1RegisterResult, 1);
    if (!wait4Answer(true, false, 200))
    {
      own_printf("ReadPage failed: no ACK\n");
      return -1;
    }

    if (ACK_frame[0] != 0x00)
    {
      own_printf("Read error: %s\n", getResultString(ACK_frame[0]));
      return -1;
    }
  }
  else
  {
    if (!ACK_received)
      own_printf("No ACK for read command\n");

    if (!ASYNC_received)
      own_printf("No ASYNC for read block command\n");

    return -1;
  }

  if (readFromRfidMemory(dataAddr + 32, blocks2read * 4))
    return -1;


  own_printf("Data bytes:");
  for (i = 0; i < blocks2read*4; i++)
    own_printf("%02X", ACK_frame[i]);
  own_printf("\n");

  return 0;
}

/**
    @brief Function used to write pages from tag
    @param[in] cblock_num - start block
    @param[in] cblocks2read - number of blocks to read
    @param[in] cdataAddr - address to store read data
    @return 0 - Success, -1 - Error
*/
int writePage(char *cblock_num, char *cblocks2write,char *cdataAddr)
{
  int blockNum = atoi(cblock_num);
  int blocks2write = atoi(cblocks2write);
  int dataAddr = atoi(cdataAddr);
  int i;
  uint8_t cmd[30];

  if (blockNum + blocks2write > 230)
  {
    own_printf("You trying to write out tag memory\n");
    return -1;
  }

  if (blocks2write * 4 + dataAddr > 256)
  {
    own_printf("You trying to write out of data buffer\n");
    return -1;
  }


  own_printf("Writing %d pages, starting from %d. Writing data from data buffer address %d.\n", blockNum, blocks2write, dataAddr);

  cmd[0] = B1RFIDCommandWritePage;
  cmd[1] = blockNum;
  cmd[2] = blocks2write;
  cmd[3] = dataAddr;

  B1Interface.SendWriteToRFIDMemoryCommand(&B1Object, B1RegisterCommand, cmd, 11);

  if (wait4Answer(true, true, 200) && (ASYNC_byte & B1AsyncPacketParamBit_RFIDCommand))
  {
    //read the status
    B1Interface.SendReadFromRFIDMemoryCommand(&B1Object, B1RegisterResult, 1);
    if (!wait4Answer(true, false, 200))
    {
      own_printf("WritePage failed: no ACK\n");
      return -1;
    }

    if (ACK_frame[0] != 0x00)
    {
      own_printf("Write error: %s\n", getResultString(ACK_frame[0]));
      return -1;
    }
  }
  else
  {
    if (!ACK_received)
      own_printf("No ACK for write command\n");

    if (!ASYNC_received)
      own_printf("No ASYNC for write block command\n");

    return -1;
  }

  return 0;
}

/**
    @brief Function used to read value from tag
    @param[in] cblock_num - start block
    @param[in] cdataAddr - address to store read data
    @param[in] ckeyAB - type of key A/B
    @param[in] ckey - key used to read memory
    @return 0 - Success, -1 - Error
*/
int readValue(char *cblock_num, char *cdataAddr, char *ckeyAB, char *ckey)
{
  int blockNum = atoi(cblock_num);
  int dataAddr = atoi(cdataAddr);
  char keyAB = *ckeyAB;
  int keyNum = -1, i;
  uint8_t cmd[30];
  unsigned int key_uint;
  uint8_t key_uint8[6];
  uint32_t value;

  if (dataAddr > 256)
  {
    own_printf("You trying to write out of data buffer\n");
    return -1;
  }

  if (keyAB != 'A' && keyAB != 'B')
  {
    own_printf("Please provide correct key A or B\n");
    return -1;
  }

  if (strlen(ckey) <= 2)
  {
    //using build in key
    keyNum = atoi(ckey);
    if (keyNum < 0 || keyNum > 39)
    {
      own_printf( "\nKey numer must be in range [0-39]!\n");
      return -1;
    }
  }
  else if (strlen(ckey) == 12)
  {
    for (i = 0; i < 6; i++)
    {
      if (sscanf(&ckey[i*2], "%02x", &key_uint) != 1)
      {
        own_printf( "\nOnly HEX chars are allowed from [0-F]!\n");
        return -1;
      }
      else
        key_uint8[i] = key_uint;
    }
  }
  else
  {
    own_printf("You should enter key as key number from module memory [range 0-39], or 12 digit hex string\n");
    return -1;
  }

  own_printf("Reading from block %d. Result will be stored in data buffer at address %d.\n", blockNum, dataAddr);

  cmd[0] = B1RFIDCommandReadValue;
  cmd[1] = blockNum;
  cmd[2] = dataAddr;

  if (keyNum != -1)
  {
    own_printf("Using key %c (key number %s stored in module memory)\n", keyAB, ckey);
    cmd[3] = ((uint8_t) keyNum);
  }
  else
  {
    own_printf("Using key %c (key %s)\n", keyAB, ckey);
    cmd[3] = 1 << 6;
    memcpy(&cmd[4], key_uint8, 6);
  }

  if (keyAB == 'B')
    cmd[3] |= 1 << 7;

  B1Interface.SendWriteToRFIDMemoryCommand(&B1Object, B1RegisterCommand, cmd, 10);

  if (wait4Answer(true, true, 200) && (ASYNC_byte & B1AsyncPacketParamBit_RFIDCommand))
  {
    //read the status
    B1Interface.SendReadFromRFIDMemoryCommand(&B1Object, B1RegisterResult, 1);
    if (!wait4Answer(true, false, 200))
    {
      own_printf("B1RFIDCommandReadValue failed: no ACK\n");
      return -1;
    }

    if (ACK_frame[0] != 0x00)
    {
      own_printf("Write error: %s\n", getResultString(ACK_frame[0]));
      return -1;
    }
  }
  else
  {
    if (!ACK_received)
      own_printf("No ACK for write command\n");

    if (!ASYNC_received)
      own_printf("No ASYNC for write block command\n");

    return -1;
  }

  if (readFromRfidMemory(dataAddr + 32, 4))
    return -1;

  value = (ACK_frame[3] << 24) | (ACK_frame[2] << 16) | (ACK_frame[1] << 8) | ACK_frame[0];

  own_printf("Value: %u\n", value);

  return 0;
}

/**
    @brief Function used to write value to tag
    @param[in] cblock_num - start block
    @param[in] cdataAddr - address to store read data
    @param[in] ckeyAB - type of key A/B
    @param[in] ckey - key used to read memory
    @return 0 - Success, -1 - Error
*/
int writeValue(char *cblock_num, char *cvalue, char *csb_addr, char *ckeyAB, char *ckey)
{
  int blockNum = atoi(cblock_num);
  uint32_t value = atoi(cvalue);
  int sb_value = atoi(csb_addr);
  char keyAB = *ckeyAB;
  int keyNum = -1, i;
  uint8_t cmd[30];
  unsigned int key_uint;
  uint8_t key_uint8[6];

  if (keyAB != 'A' && keyAB != 'B')
  {
    own_printf("Please provide correct key A or B\n");
    return -1;
  }

  if (strlen(ckey) <= 2)
  {
    //using build in key
    keyNum = atoi(ckey);
    if (keyNum < 0 || keyNum > 39)
    {
      own_printf( "\nKey numer must be in range [0-39]!\n");
      return -1;
    }
  }
  else if (strlen(ckey) == 12)
  {
    for (i = 0; i < 6; i++)
    {
      if (sscanf(&ckey[i*2], "%02x", &key_uint) != 1)
      {
        own_printf( "\nOnly HEX chars are allowed from [0-F]!\n");
        return -1;
      }
      else
        key_uint8[i] = key_uint;
    }
  }
  else
  {
    own_printf("You should enter key as key number from module memory [range 0-39], or 12 digit hex string\n");
    return -1;
  }

  own_printf("Writing value to block %d\n", blockNum);

  cmd[0] = B1RFIDCommandWriteValue;
  cmd[1] = blockNum;
  cmd[2] = value & 0xff;
  cmd[3] = (value >> 8) & 0xff;
  cmd[4] = (value >> 16) & 0xff;
  cmd[5] = (value >> 24) & 0xff;
  cmd[6] = sb_value;

  if (keyNum != -1)
  {
    own_printf("Using key %c (key number %s stored in module memory)\n", keyAB, ckey);
    cmd[7] = ((uint8_t) keyNum);
  }
  else
  {
    own_printf("Using key %c (key %s)\n", keyAB, ckey);
    cmd[7] = 1 << 6;
    memcpy(&cmd[8], key_uint8, 6);
  }

  if (keyAB == 'B')
    cmd[7] |= 1 << 7;

  B1Interface.SendWriteToRFIDMemoryCommand(&B1Object, B1RegisterCommand, cmd, 14);

  if (wait4Answer(true, true, 200) && (ASYNC_byte & B1AsyncPacketParamBit_RFIDCommand))
  {
    //read the status
    B1Interface.SendReadFromRFIDMemoryCommand(&B1Object, B1RegisterResult, 1);
    if (!wait4Answer(true, false, 200))
    {
      own_printf("B1RFIDCommandWriteValue failed: no ACK\n");
      return -1;
    }

    if (ACK_frame[0] != 0x00)
    {
      own_printf("Write error: %s\n", getResultString(ACK_frame[0]));
      return -1;
    }
  }
  else
  {
    if (!ACK_received)
      own_printf("No ACK for write command\n");

    if (!ASYNC_received)
      own_printf("No ASYNC for write block command\n");

    return -1;
  }

  return 0;
}

/**
    @brief Function used to increment value stored in tag memory
    @param[in] cblock_num - block with value
    @param[in] cdelta - value to increment data
    @param[in] ckeyAB - type of key A/B
    @param[in] ckey - key used to read memory
    @return 0 - Success, -1 - Error
*/
int incrementValue(char *cblock_num, char *cdelta, char *ckeyAB, char *ckey, bool negative)
{
  int blockNum = atoi(cblock_num);
  uint32_t delta = atoi(cdelta);
  char keyAB = *ckeyAB;
  int keyNum = -1, i;
  uint8_t cmd[30];
  unsigned int key_uint;
  uint8_t key_uint8[6];

  if (keyAB != 'A' && keyAB != 'B')
  {
    own_printf("Please provide correct key A or B\n");
    return -1;
  }

  if (strlen(ckey) <= 2)
  {
    //using build in key
    keyNum = atoi(ckey);
    if (keyNum < 0 || keyNum > 39)
    {
      own_printf( "\nKey numer must be in range [0-39]!\n");
      return -1;
    }
  }
  else if (strlen(ckey) == 12)
  {
    for (i = 0; i < 6; i++)
    {
      if (sscanf(&ckey[i*2], "%02x", &key_uint) != 1)
      {
        own_printf( "\nOnly HEX chars are allowed from [0-F]!\n");
        return -1;
      }
      else
        key_uint8[i] = key_uint;
    }
  }
  else
  {
    own_printf("You should enter key as key number from module memory [range 0-39], or 12 digit hex string\n");
    return -1;
  }

  own_printf("Modify value at block %d\n", blockNum);

  if (negative)
    cmd[0] = B1RFIDCommandDecrementValue;
  else
    cmd[0] = B1RFIDCommandIncrementValue;
  cmd[1] = blockNum;
  cmd[2] = delta & 0xff;
  cmd[3] = (delta >> 8) & 0xff;
  cmd[4] = (delta >> 16) & 0xff;
  cmd[5] = (delta >> 24) & 0xff;

  if (keyNum != -1)
  {
    own_printf("Using key %c (key number %s stored in module memory)\n", keyAB, ckey);
    cmd[6] = ((uint8_t) keyNum);
  }
  else
  {
    own_printf("Using key %c (key %s)\n", keyAB, ckey);
    cmd[6] = 1 << 6;
    memcpy(&cmd[7], key_uint8, 6);
  }

  if (keyAB == 'B')
    cmd[6] |= 1 << 7;

  B1Interface.SendWriteToRFIDMemoryCommand(&B1Object, B1RegisterCommand, cmd, 13);

  if (wait4Answer(true, true, 200) && (ASYNC_byte & B1AsyncPacketParamBit_RFIDCommand))
  {
    //read the status
    B1Interface.SendReadFromRFIDMemoryCommand(&B1Object, B1RegisterResult, 1);
    if (!wait4Answer(true, false, 200))
    {
      if (negative)
        own_printf("B1RFIDCommandDecrementValue failed: no ACK\n");
      else
        own_printf("B1RFIDCommandIncrementValue failed: no ACK\n");

      return -1;
    }

    if (ACK_frame[0] != 0x00)
    {
      own_printf("Write error: %s\n", getResultString(ACK_frame[0]));
      return -1;
    }
  }
  else
  {
    if (!ACK_received)
      own_printf("No ACK for write command\n");

    if (!ASYNC_received)
      own_printf("No ASYNC for write block command\n");

    return -1;
  }

  return 0;
}

/**
    @brief Function used to read/write value stored in tag memory
    @param[in] cblock_num - block with value
    @param[in] ckeyAB - type of key A/B
    @param[in] ckey - key used to read memory
    @param[in] command - command put/get
    @return 0 - Success, -1 - Error
*/
int putgetValue(char *cblock_num, char *ckeyAB, char *ckey, uint8_t command)
{
  int blockNum = atoi(cblock_num);
  char keyAB = *ckeyAB;
  int keyNum = -1, i;
  uint8_t cmd[30];
  unsigned int key_uint;
  uint8_t key_uint8[6];

  if (keyAB != 'A' && keyAB != 'B')
  {
    own_printf("Please provide correct key A or B\n");
    return -1;
  }

  if (strlen(ckey) <= 2)
  {
    //using build in key
    keyNum = atoi(ckey);
    if (keyNum < 0 || keyNum > 39)
    {
      own_printf( "\nKey numer must be in range [0-39]!\n");
      return -1;
    }
  }
  else if (strlen(ckey) == 12)
  {
    for (i = 0; i < 6; i++)
    {
      if (sscanf(&ckey[i*2], "%02x", &key_uint) != 1)
      {
        own_printf( "\nOnly HEX chars are allowed from [0-F]!\n");
        return -1;
      }
      else
        key_uint8[i] = key_uint;
    }
  }
  else
  {
    own_printf("You should enter key as key number from module memory [range 0-39], or 12 digit hex string\n");
    return -1;
  }

  own_printf("Modify value at block %d\n", blockNum);

  cmd[0] = command;
  cmd[1] = blockNum;

  if (keyNum != -1)
  {
    own_printf("Using key %c (key number %s stored in module memory)\n", keyAB, ckey);
    cmd[2] = ((uint8_t) keyNum);
  }
  else
  {
    own_printf("Using key %c (key %s)\n", keyAB, ckey);
    cmd[2] = 1 << 6;
    memcpy(&cmd[3], key_uint8, 6);
  }

  if (keyAB == 'B')
    cmd[6] |= 1 << 7;

  B1Interface.SendWriteToRFIDMemoryCommand(&B1Object, B1RegisterCommand, cmd, 9);

  if (wait4Answer(true, true, 200) && (ASYNC_byte & B1AsyncPacketParamBit_RFIDCommand))
  {
    //read the status
    B1Interface.SendReadFromRFIDMemoryCommand(&B1Object, B1RegisterResult, 1);
    if (!wait4Answer(true, false, 200))
    {
      if (command == B1RFIDCommandTransferValue)
        own_printf("B1RFIDCommandTransferValue failed: no ACK\n");
      else
        own_printf("B1RFIDCommandRestoreValue failed: no ACK\n");
      return -1;
    }

    if (ACK_frame[0] != 0x00)
    {
      own_printf("Write error: %s\n", getResultString(ACK_frame[0]));
      return -1;
    }
  }
  else
  {
    if (!ACK_received)
      own_printf("No ACK for write command\n");

    if (!ASYNC_received)
      own_printf("No ASYNC for write block command\n");

    return -1;
  }

  return 0;
}

/**
    @brief Function used to tag version
    @return 0 - Success, -1 - Error
*/
int getVersion()
{
  uint8_t cmd[5],k;

  cmd[0] = B1RFIDCommandGetVersion;

  B1Interface.SendWriteToRFIDMemoryCommand(&B1Object, B1RegisterCommand, cmd, 1);

  if (wait4Answer(true, true, 200) && (ASYNC_byte & B1AsyncPacketParamBit_RFIDCommand))
  {
    //read the status
    B1Interface.SendReadFromRFIDMemoryCommand(&B1Object, B1RegisterResult, 1);
    if (!wait4Answer(true, false, 200))
    {
      own_printf("B1RFIDCommandGetVersion failed: no ACK\n");
      return -1;
    }

    if (ACK_frame[0] != 0x00)
    {
      own_printf("Write error: %s (probably this tag don't support getVersion command)\n", getResultString(ACK_frame[0]));
      return -1;
    }
  }
  else
  {
    if (!ACK_received)
      own_printf("No ACK for write command\n");

    if (!ASYNC_received)
      own_printf("No ASYNC for write block command\n");

    return -1;
  }

  if (readFromRfidMemory(B1RegisterDataBuffer, 8))
    return -1;

  own_printf("Version bytes:");
  for (k=0; k< 8; k++)
    own_printf("%02X", ACK_frame[k]);
  own_printf("\n");

  return 0;
}

/**
    @brief Function used to read tag signature
    @return 0 - Success, -1 - Error
*/
int readSignature()
{
  uint8_t cmd[5],k;

  cmd[0] = B1RFIDCommandReadSignature;

  B1Interface.SendWriteToRFIDMemoryCommand(&B1Object, B1RegisterCommand, cmd, 1);

  if (wait4Answer(true, true, 200) && (ASYNC_byte & B1AsyncPacketParamBit_RFIDCommand))
  {
    //read the status
    B1Interface.SendReadFromRFIDMemoryCommand(&B1Object, B1RegisterResult, 1);
    if (!wait4Answer(true, false, 200))
    {
      own_printf("ReadSignature failed: no ACK\n");
      return -1;
    }

    if (ACK_frame[0] != 0x00)
    {
      own_printf("Write error: %s (probably this tag don't support read signature command)\n", getResultString(ACK_frame[0]));
      return -1;
    }
  }
  else
  {
    if (!ACK_received)
      own_printf("No ACK for write command\n");

    if (!ASYNC_received)
      own_printf("No ASYNC for write block command\n");

    return -1;
  }

  if (readFromRfidMemory(B1RegisterDataBuffer, 8))
    return -1;

  own_printf("Signature bytes:");
  for (k=0; k< 32; k++)
    own_printf("%02X", ACK_frame[k]);
  own_printf("\n");

  return 0;
}


/**
    @brief Function used to configure tag UID
    @param[in] cuidType - UID type
    @param[in] ckeyAB - type of key A/B
    @param[in] ckey - key used to write configuration
    @return 0 - Success, -1 - Error
*/
int cfgUid(char *cuidType, char *ckeyAB, char *ckey)
{
  int uidType = atoi(cuidType);
  char keyAB = *ckeyAB;
  unsigned int key_uint;
  uint8_t key_uint8[6];
  int keyNum = -1, i;
  uint8_t cmd[30];

  if (keyAB != 'A' && keyAB != 'B')
  {
    own_printf("Please provide correct key A or B\n");
    return -1;
  }

  if (strlen(ckey) <= 2)
  {
    //using build in key
    keyNum = atoi(ckey);
    if (keyNum < 0 || keyNum > 39)
    {
      own_printf( "\nKey numer must be in range [0-39]!\n");
      return -1;
    }
  }
  else if (strlen(ckey) == 12)
  {
    for (i = 0; i < 6; i++)
    {
      if (sscanf(&ckey[i*2], "%02x", &key_uint) != 1)
      {
        own_printf( "\nOnly HEX chars are allowed from [0-F]!\n");
        return -1;
      }
      else
        key_uint8[i] = key_uint;
    }
  }
  else
  {
    own_printf("You should enter key as key number from module memory [range 0-39], or 12 digit hex string\n");
    return -1;
  }

  cmd[0] = B1RFIDCommandConfigureUID;
  cmd[1] = uidType;

  if (keyNum != -1)
  {
    own_printf("Using key %c (key number %s stored in module memory)\n", keyAB, ckey);
    cmd[2] = ((uint8_t) keyNum);
  }
  else
  {
    own_printf("Using key %c (key %s)\n", keyAB, ckey);
    cmd[2] = 1 << 6;
    memcpy(&cmd[3], key_uint8, 6);
  }

  if (keyAB == 'B')
    cmd[2] |= 1 << 7;

  B1Interface.SendWriteToRFIDMemoryCommand(&B1Object, B1RegisterCommand, cmd, 9);

  if (wait4Answer(true, true, 200) && (ASYNC_byte & B1AsyncPacketParamBit_RFIDCommand))
  {
    //read the status
    B1Interface.SendReadFromRFIDMemoryCommand(&B1Object, B1RegisterResult, 1);
    if (!wait4Answer(true, false, 200))
    {
      own_printf("Configure UID failed: no ACK\n");
      return -1;
    }

    if (ACK_frame[0] != 0x00)
    {
      own_printf("Configure UID error: %s\n", getResultString(ACK_frame[0]));
      return -1;
    }
  }
  else
  {
    if (!ACK_received)
      own_printf("No ACK for read command\n");

    if (!ASYNC_received)
      own_printf("No ASYNC for read block command\n");

    return -1;
  }

  return 0;
}

/**
    @brief Function used to read counter from tag
    @param[in] ccounterNum - counter number
    @param[in] cdataAddr - address in module memory to store result
    @return 0 - Success, -1 - Error
*/
int readCounter(char *ccounterNum, char *cdataAddr)
{
  int counterNum = atoi(ccounterNum);
  int dataAddr = atoi(cdataAddr);
  uint8_t cmd[30];
  uint32_t value;

  if (dataAddr > 256)
  {
    own_printf("You trying to write out of data buffer\n");
    return -1;
  }

  own_printf("Reading counter %d. Result will be stored in data buffer at address %d. \n", counterNum, dataAddr);

  cmd[0] = B1RFIDCommandReadCounter;
  cmd[1] = counterNum;
  cmd[2] = dataAddr;

  B1Interface.SendWriteToRFIDMemoryCommand(&B1Object, B1RegisterCommand, cmd, 3);

  if (wait4Answer(true, true, 200) && (ASYNC_byte & B1AsyncPacketParamBit_RFIDCommand))
  {
    //read the status
    B1Interface.SendReadFromRFIDMemoryCommand(&B1Object, B1RegisterResult, 1);
    if (!wait4Answer(true, false, 200))
    {
      own_printf("Read counter failed: no ACK\n");
      return -1;
    }

    if (ACK_frame[0] != 0x00)
    {
      own_printf("Write error: %s\n", getResultString(ACK_frame[0]));
      return -1;
    }
  }
  else
  {
    if (!ACK_received)
      own_printf("No ACK for write command\n");

    if (!ASYNC_received)
      own_printf("No ASYNC for write block command\n");

    return -1;
  }

  if (readFromRfidMemory(dataAddr + 32, 2))
    return -1;

  value = (ACK_frame[2] << 16) | (ACK_frame[1] << 8) | ACK_frame[0];

  own_printf("Counter value: %u\n", value);

  return 0;
}

/**
    @brief Function used to increment counter from tag
    @param[in] ccounterNum - counter number
    @param[in] cdelta - delta value
    @return 0 - Success, -1 - Error
*/
int incrementCounter(char *ccounterNum, char *cdelta)
{
  int counterNum = atoi(ccounterNum);
  uint32_t delta = atoi(cdelta);
  uint8_t cmd[30];

  own_printf("Increment counter %d by %d\n", counterNum, delta);

  cmd[0] = B1RFIDCommandIncrementCounter;
  cmd[1] = delta & 0xff;
  cmd[2] = (delta >> 8) & 0xff;
  cmd[3] = (delta >> 16) & 0xff;

  B1Interface.SendWriteToRFIDMemoryCommand(&B1Object, B1RegisterCommand, cmd, 4);

  if (wait4Answer(true, true, 200) && (ASYNC_byte & B1AsyncPacketParamBit_RFIDCommand))
  {
    //read the status
    B1Interface.SendReadFromRFIDMemoryCommand(&B1Object, B1RegisterResult, 1);
    if (!wait4Answer(true, false, 200))
    {
      own_printf("IncrementCounter failed: no ACK\n");

      return -1;
    }

    if (ACK_frame[0] != 0x00)
    {
      own_printf("IncrementCounter error: %s\n", getResultString(ACK_frame[0]));
      return -1;
    }
  }
  else
  {
    if (!ACK_received)
      own_printf("No ACK for write command\n");

    if (!ASYNC_received)
      own_printf("No ASYNC for write block command\n");

    return -1;
  }

  return 0;
}

/**
    @brief Function used check event
    @param[in] cnumber - event number
    @param[in] cdataAddr - address in module memory to store result
    @return 0 - Success, -1 - Error
*/
int checkEvent(char *cnumber, char *cdataAddr)
{
  int number = atoi(cnumber);
  int dataAddr = atoi(cdataAddr);
  uint8_t cmd[30];
  uint32_t value;

  if (dataAddr > 256)
  {
    own_printf("You trying to write out of data buffer\n");
    return -1;
  }

  own_printf("Reading tearing event %d. Result will be stored in data buffer at address %d. \n", number, dataAddr);

  cmd[0] = B1RFIDCommandCheckTearingEvent;
  cmd[1] = number;
  cmd[2] = dataAddr;

  B1Interface.SendWriteToRFIDMemoryCommand(&B1Object, B1RegisterCommand, cmd, 3);

  if (wait4Answer(true, true, 200) && (ASYNC_byte & B1AsyncPacketParamBit_RFIDCommand))
  {
    //read the status
    B1Interface.SendReadFromRFIDMemoryCommand(&B1Object, B1RegisterResult, 1);
    if (!wait4Answer(true, false, 200))
    {
      own_printf("Read tearing event failed: no ACK\n");
      return -1;
    }

    if (ACK_frame[0] != 0x00)
    {
      own_printf("Read tearing event error: %s\n", getResultString(ACK_frame[0]));
      return -1;
    }
  }
  else
  {
    if (!ACK_received)
      own_printf("No ACK for write command\n");

    if (!ASYNC_received)
      own_printf("No ASYNC for write block command\n");

    return -1;
  }

  if (readFromRfidMemory(dataAddr + 32, 1))
    return -1;

  value = ACK_frame[0];

  own_printf("Event value: %u\n", value);

  return 0;
}

/**
    @brief Function used for password authorization
    @param[in] cdataAddr - data address
    @param[in] ckey - key used to authorize or key number
    @return 0 - Success, -1 - Error
*/
int passwdAuth(char *cdataAddr, char *ckey)
{
  int dataAddr = atoi(cdataAddr);

  unsigned int key_uint;
  uint8_t key_uint8[6];
  int keyNum = -1, i;
  uint8_t cmd[30];

  if (dataAddr+2 > 256)
  {
    own_printf("You trying to read out of data buffer\n");
    return -1;
  }

  if (strlen(ckey) <= 2)
  {
    //using build in key
    keyNum = atoi(ckey);
    if (keyNum < 0 || keyNum > 39)
    {
      own_printf( "\nKey numer must be in range [0-39]!\n");
      return -1;
    }
  }
  else if (strlen(ckey) == 8)
  {
    for (i = 0; i < 4; i++)
    {
      if (sscanf(&ckey[i*2], "%02x", &key_uint) != 1)
      {
        own_printf( "\nOnly HEX chars are allowed from [0-F]!\n");
        return -1;
      }
      else
        key_uint8[i] = key_uint;
    }
  }
  else
  {
    own_printf("You should enter key as key number from module memory [range 0-39], or 8 digit hex string\n");
    return -1;
  }

  cmd[0] = B1RFIDCommandPasswordAuthentication;
  cmd[1] = dataAddr;

  if (keyNum != -1)
  {
    own_printf("Using password %d stored in module memory)\n", keyNum);
    cmd[2] = ((uint8_t) keyNum);
  }
  else
  {
    own_printf("Using password %s.\n", ckey);
    cmd[2] = 0x80;
    memcpy(&cmd[3], key_uint8, 4);
  }

  B1Interface.SendWriteToRFIDMemoryCommand(&B1Object, B1RegisterCommand, cmd, 7);

  if (wait4Answer(true, true, 200) && (ASYNC_byte & B1AsyncPacketParamBit_RFIDCommand))
  {
    //read the status
    B1Interface.SendReadFromRFIDMemoryCommand(&B1Object, B1RegisterResult, 1);
    if (!wait4Answer(true, false, 200))
    {
      own_printf("Password authentication failed: no ACK\n");
      return -1;
    }

    if (ACK_frame[0] != 0x00)
    {
      own_printf("Password authentication error: %s\n", getResultString(ACK_frame[0]));
      return -1;
    }
  }
  else
  {
    if (!ACK_received)
      own_printf("No ACK for read command\n");

    if (!ASYNC_received)
      own_printf("No ASYNC for read block command\n");

    return -1;
  }

  if (readFromRfidMemory(dataAddr + 32, 2))
    return -1;


  own_printf("Result:");

  for (i = 0; i < 2; i++)
    own_printf("%02X", ACK_frame[i]);
  own_printf("\n");

  return 0;
}

/**
    @brief Function used for polling configuration
    @params - please read manual for parameters description
    @return 0 - Success, -1 - Error
*/
int pollingEnable(char *cPeriod, char *cNumberOfTags,
	char *cDefAsyncPacket, char *cDefIOConfig, char *cDefPWMIO, char *cDefPWMDuty, char *cDefPWMPeriod, char *cDefTimeout,
	char *cUndefAsyncPacket, char *cUndefIOConfig, char *cUndefPWMIO, char *cUndefPWMDuty, char *cUndefPWMPeriod, char *cUndefTimeout)
{
  uint8_t period = atoi(cPeriod);
  uint8_t numberOfTags = atoi(cNumberOfTags);

  uint8_t defAsyncPacket = atoi(cDefAsyncPacket);
  uint8_t defIOConfig = atoi(cDefIOConfig);
  uint8_t defPWMIO = atoi(cDefPWMIO);
  uint8_t defPWMDuty = atoi(cDefPWMDuty);
  uint32_t defPWMPeriod = atoi(cDefPWMPeriod);
  uint8_t defTimeout = atoi(cDefTimeout);

  uint8_t undefAsyncPacket = atoi(cUndefAsyncPacket);
  uint8_t undefIOConfig = atoi(cUndefIOConfig);
  uint8_t undefPWMIO = atoi(cUndefPWMIO);
  uint8_t undefPWMDuty = atoi(cUndefPWMDuty);
  uint32_t undefPWMPeriod = atoi(cUndefPWMPeriod);
  uint8_t undefTimeout = atoi(cUndefTimeout);

  uint8_t cmd[30];

  cmd[0] = B1RFIDCommandPolling;
  cmd[1] = period;
  cmd[2] = numberOfTags;
  cmd[3] = defAsyncPacket;
  cmd[4] = defIOConfig;
  cmd[5] = defPWMIO;
  cmd[6] = defPWMDuty;
  cmd[7] = defPWMPeriod & 0xff;
  cmd[8] = (defPWMPeriod >> 8) & 0xff;
  cmd[9] = (defPWMPeriod >> 16) & 0xff;
  cmd[10] = defTimeout;

  cmd[11] = undefAsyncPacket;
  cmd[12] = undefIOConfig;
  cmd[13] = undefPWMIO;
  cmd[14] = undefPWMDuty;
  cmd[15] = undefPWMPeriod & 0xff;
  cmd[16] = (undefPWMPeriod >> 8) & 0xff;
  cmd[17] = (undefPWMPeriod >> 16) & 0xff;
  cmd[18] = undefTimeout;


  B1Interface.SendWriteToRFIDMemoryCommand(&B1Object, B1RegisterCommand, cmd, 19);

  if (wait4Answer(true, true, 200) && (ASYNC_byte & B1AsyncPacketParamBit_RFIDCommand))
  {
    //read the status
    B1Interface.SendReadFromRFIDMemoryCommand(&B1Object, B1RegisterResult, 1);
    if (!wait4Answer(true, false, 200))
    {
      own_printf("Polling command failed: no ACK\n");
      return -1;
    }

    if (ACK_frame[0] != 0x00)
    {
      own_printf("Polling command failed: %s\n", getResultString(ACK_frame[0]));
      return -1;
    }
  }
  else
  {
    if (!ACK_received)
      own_printf("No ACK for read command\n");

    if (!ASYNC_received)
      own_printf("No ASYNC for read block command\n");

    return -1;
  }

  return 0;
}

/**
    @brief Function used to enable PWM on selected GPIO
    @param[in] gpio - gpio number [0-2]
    @param[in] duty - PWM duty
    @param[in] hz - hertz number
    @return 0 - Success, -1 - Error
*/
int enablePWM(char *cgpio, char *chz, char *cduty)
{
  int gpio = atoi(cgpio);
  int duty = atoi(cduty);
  uint32_t hz = atoi(chz);
  uint8_t cmd[30];

  if (gpio < 0 || gpio > 3)
  {
    own_printf("GPIO must be in range 0-2\n");
    return -1;
  }

  if (duty < 1 || duty > 99)
  {
    own_printf("Duty must be in range 1-99\n");
    return -1;
  }

  own_printf("Enabling PWM on GPIO %d, HZ=%d, duty=%d\n", gpio, hz, duty);

  B1Interface.SendEnablePWMCommand(&B1Object,PWMConfigFormatUint32_Hz, &hz, gpio, duty);

  if (!wait4Answer(true, false, 200))
  {
    own_printf("Enabling PWM failed: no ACK\n");
    return -1;
  }

  return 0;
}

/**
    @brief Function used to toggle GPI state
    @param[in] gpio - gpio number [0-3]
    @param[in] value - can be 0 or 1
    @return 0 - Success, -1 - Error
*/
int setGPIO(char *cgpio, char *cvalue)
{
  int gpio = atoi(cgpio);
  int value = atoi(cvalue);
  uint8_t cmd[30];

  if (gpio < 0 || gpio > 3)
  {
    own_printf("GPIO must be in range 0-3\n");
    return -1;
  }

  if (value < 0 || value > 1)
  {
    own_printf("Value must be 0 or 1\n");
    return -1;
  }

  own_printf("Set state %d on GPIO %d\n", value, gpio);

  B1Interface.SendSetIOStateCommand(&B1Object,gpio, GpioStateOutputLow+value);

  if (!wait4Answer(true, false, 200))
  {
    own_printf("Set state failed: no ACK\n");
    return -1;
  }

  return 0;
}

/**
    @brief Function prints help
*/
void print_usage()
{
  own_printf( "\nUsage: rfidb1-tool [device path] [command] [args ...]\n");
  own_printf( "Available commands:\n");
  own_printf( "\n---------Commands for RFID B1 memory---------\n");
  own_printf( " unlock [PASSWORD]   - unlock RFID B1 memory used for AES keys, passwords, user data etc.\n");
  own_printf( "                       password should be 8-bytes long string in hex format like 1122334455667788\n");
  own_printf( " lock                - lock RFID B1 memory used for AES keys, passwords, user data etc.\n");
  own_printf( " passwd [PASSWORD]   - write new password for B1 module (module need to be unlocked).\n");
  own_printf( " reset_to_defaults [MODE]  - Reset module to default settings (0- software via UART command, 1 - hardware using RST/PWR_DOWN pins, works only on Raspberry-Pi HAT).\n");
  own_printf( " getmoduleversion    - get module firmware version string\n");

  own_printf( " rkey [X]            - read key number X from module memory, output is HEX string (module need to be unlocked).\n");
  own_printf( " wkey [X] [KEY]      - write key number X to module memory. Key format is 112233445566 (HEX string). (module need to be unlocked).\n");

  own_printf( " raesk [X]            - read AES key number 0 or 1 from module memory, output is HEX string (module need to be unlocked).\n");
  own_printf( " waesk [X] [KEY]      - write AES key number 0 or 1 to module memory/ Key format is 11223344556677889900112233445566 (16-HEX string). (module need to be unlocked).\n");
  own_printf( " raesv [X]            - read AES vector number 0 or 1 from module memory, output is HEX string (module need to be unlocked).\n");
  own_printf( " waesv [X] [VECTOR]   - write AES vector number 0 or 1 to module memory. Vector format is 11223344556677889900112233445566 (16-HEX string). (module need to be unlocked).\n");

  own_printf( " rdata [ADDR] [LEN]  - read from data buffer in module memory.\n");
  own_printf( " wdata [ADDR] [HEX]  - write to data buffer in module memory.\n");
  own_printf( " rread [ADDR] [LEN]  - raw read from module memory.\n");
  own_printf( " rwrite [ADDR] [HEX] - raw write to module memory.\n");

  own_printf( " copy [DEST] [SRC] [LEN]  - Copy data from SRC address to DEST.\n");

  own_printf( " encrypt [KEY] [VECTOR] [OFFSET] [BLOCKS] - Encrypt data buffer using AES key and vector stored in memory. Block size is 16\n");
  own_printf( " dencrypt [KEY] [VECTOR] [OFFSET] [BLOCKS] - Decrypt data buffer using AES key and vector stored in memory. Block size is 16\n");
  own_printf( " crc [ADDR] [LENGHT] [OFFSET] - Calculate CRC from module memory (not only data buffer!), and store it in to data buffer at OFFSET\n");

  own_printf( "\n\n---------RFID commands---------\n");
  own_printf( " poll                                            - wait for UID tag and exit (CTRL+C to stop)\n");
  own_printf( " uid                                             - read UID. This command should be called before any reads/writes on card, or if any RDID command fail.\n");
  own_printf( " halt                                            - turn of RFID antenna.\n");
  own_printf( " rblock [ADDR][BLOCKS][DATA OFFSET][KEY_AB][KEY] - read block from Mifare tags\n");
  own_printf( " wblock [ADDR][BLOCKS][DATA OFFSET][KEY_AB][KEY] - write block from Mifare tags\n");
  own_printf( " rdblock [ADDR][BLOCKS][DATA OFFSET][KEY_AB][KEY]- read data block from Mifare tags (skipping sector trailers)\n");
  own_printf( " wdblock [ADDR][BLOCKS][DATA OFFSET][KEY_AB][KEY]- write data block from Mifare tags (skipping sector trailers)\n");

  own_printf( " rpage [ADDR][PAGES][DATA OFFSET]               - read data page from Mifare Ul. EV1, and NTAGx tags (skipping sector trailers)\n");
  own_printf( " wpage [ADDR][PAGES][DATA OFFSET]               - write data page from Mifare Ul. EV1, and NTAGx tags (skipping sector trailers)\n");

  own_printf( " rvalue [ADDR][OFFSET][KEY_AB][KEY]              - read value from Mifare tags\n");
  own_printf( " wvalue [ADDR][VALUE][SB_ADDR][KEY_AB][KEY]      - write value from Mifare tags\n");
  own_printf( " incvalue [ADDR][DELTA][KEY_AB][KEY]             - increment value from Mifare tags\n");
  own_printf( " decvalue [ADDR][VALUE][KEY_AB][KEY]             - decrement value from Mifare tags\n");
  own_printf( " transfervalue [ADDR][KEY_AB][KEY]               - transfer value from Mifare tags\n");
  own_printf( " restorevalue [ADDR][KEY_AB][KEY]                - restore value from Mifare tags\n");
  own_printf( " getversion                                      - restore value from Mifare tags\n");
  own_printf( " rsignature                                      - read signature from Mifare Ul. EV1, and NTAGx tags\n");
  own_printf( " cfguid [UID_TYPE][KEY_AB][KEY]                  - configure UID on some Mifare card\n");
  own_printf( " rcounter [NUM][OFFSET]                          - read counter from Mifare Ul. EV1, and NTAGx tags\n");
  own_printf( " icounter [NUM][DELTA]                           - incremet counter on Mifare Ul. EV1 tags\n");
  own_printf( " checkevent [NUM][OFFSET]                        - read teating event from Mifare Ul. EV1 tags\n");
  own_printf( " passwdauth [OFFSET][PASSWORD]                   - try to auth. with password stored in module memory [0-39], or 4 bytes hex string eg. 11223344.\n");
  own_printf( "                                                   Mifare Ul. EV1, and NTAGx tags\n");
  own_printf( " polling [PERIOD] [NumberOfTags] \n"
                        "[DefAcyncPacket]   [DefIOCfg]   [DefPWMCfg]   [DefPWMDuty]   [DefPWMPeriod]   [DefTimeout]\n"
                        "[UndefAcyncPacket] [UndefIOCfg] [UndefPWMCfg] [UndefPWMDuty] [UndefPWMPeriod] [UndefTimeout]\n");
  own_printf( "                                                 - enable internal module polling. Read manual for more informations.\n");
  own_printf( "\n---------Commands for RFID B1 on Raspberry Pi HAT only (root privileges may be required) !!!---------\n");
  own_printf( " enable              - enable RFIDB1 module.\n");
  own_printf( " disable             - disable RFIDB1 module.\n");
  own_printf( " reset               - reset RFIDB1 module.\n");

  own_printf( "\n--------- Commands for GPIO managment---------\n");
  own_printf( " pwm [gpio] [freq] [duty] - enable PWM on GPIO pin.\n");
  own_printf( " sgpio [gpio] [value]     - set GPIO state.\n");

  if (serial_fd != -1)
    close(serial_fd);
  own_printf( "\n");
  exit(EXIT_FAILURE);
}


int parseCommands(int argc, char * argv[])
{
    if (strcmp(argv[2], "enable") == 0)
    {
      if (initializeGpioOnHat())
        return -1;

      if (B1Interface.EnableModule(&B1Object) == B1StatusT_OK)
        own_printf("RFIDB1 module on Raspberry Pi HAT enabled.\n");
      else
        own_printf("Can't enable RFIDB1 module on Raspberry Pi!.\n");
    }
    else
    if (strcmp(argv[2], "disable") == 0)
    {
      if (initializeGpioOnHat())
        return -1;

      if (B1Interface.DisableModule(&B1Object) == B1StatusT_OK)
        own_printf("RFIDB1 module on Raspberry Pi HAT disabled.\n");
      else
        own_printf("Can't disable RFIDB1 module on Raspberry Pi!.\n");
    }
    else
    if (strcmp(argv[2], "reset") == 0)
    {
      if (initializeGpioOnHat())
        return -1;

      if (B1Interface.HardResetModule(&B1Object) == B1StatusT_OK)
        own_printf("RFIDB1 module on Raspberry Pi HAT reset.\n");
      else
        own_printf("Can't reset RFIDB1 module on Raspberry Pi!.\n");
    }
    else
    if (strcmp(argv[2], "poll") == 0)
    {
      uint8_t tagType, uid[32], uid_len;

      own_printf("\nWaiting for tag (CTRL+C to abort)...\n");

      while (1)
      {
        tagType = getUid(uid, &uid_len);
        if (tagType != TagType_NoTag)
        {
          print_tag(tagType, uid, uid_len);
          own_printf("\nWaiting for tag (CTRL+C to abort)...\n");
          sleep(1);
        }
      }
    }
    else
    if (strcmp(argv[2], "uid") == 0)
    {
      uint8_t tagType, uid[32], uid_len;

      tagType = getUid(uid, &uid_len);

      print_tag(tagType, uid, uid_len);
    }
    else
    if (strcmp(argv[2], "halt") == 0)
    {
      return halt();
    }
    else
    if (strcmp(argv[2], "unlock") == 0 && argc == 4)
    {
      return unlockModule(argv[3]);
    }
    else
    if (strcmp(argv[2], "lock") == 0 && argc == 3)
    {
      return lockModule();
    }
    else
    if (strcmp(argv[2], "reset_to_defaults") == 0 && argc == 4)
    {
      return reset2Defaults(argv[3]);
    }
    else
    if (strcmp(argv[2], "getmoduleversion") == 0)
    {
      return getModuleVersion();
    }
    else
    if (strcmp(argv[2], "passwd") == 0 && argc == 4)
    {
      return changePassword(argv[3]);
    }
    else
    if (strcmp(argv[2], "rkey") == 0 && argc == 4)
    {
      return readKey(argv[3]);
    }
    else
    if (strcmp(argv[2], "wkey") == 0 && argc == 5)
    {
      return writeKey(argv[3],argv[4]);
    }
    else
    if (strcmp(argv[2], "raesk") == 0 && argc == 4)
    {
      return readAesKey(argv[3]);
    }
    else
    if (strcmp(argv[2], "waesk") == 0 && argc == 5)
    {
      return writeAesKey(argv[3],argv[4]);
    }
    else
    if (strcmp(argv[2], "raesv") == 0 && argc == 4)
    {
      return readAesVector(argv[3]);
    }
    else
    if (strcmp(argv[2], "waesv") == 0 && argc == 5)
    {
      return writeAesVector(argv[3],argv[4]);
    }
    else
    if (strcmp(argv[2], "rdata") == 0 && argc == 5)
    {
      return readData(argv[3],argv[4]);
    }
    else
    if (strcmp(argv[2], "wdata") == 0 && argc == 5)
    {
      return writeData(argv[3],argv[4]);
    }
    else
    if (strcmp(argv[2], "rread") == 0 && argc == 5)
    {
      return rawRead(argv[3],argv[4]);
    }
    else
    if (strcmp(argv[2], "rwrite") == 0 && argc == 5)
    {
      return rawWrite(argv[3],argv[4]);
    }
    else
    if (strcmp(argv[2], "copy") == 0 && argc == 6)
    {
      return copyData(argv[3],argv[4],argv[5]);
    }
    else
    if (strcmp(argv[2], "encrypt") == 0 && argc == 7)
    {
      return cryptData(argv[3],argv[4],argv[5],argv[6], false);
    }
    else
    if (strcmp(argv[2], "decrypt") == 0 && argc == 7)
    {
      return cryptData(argv[3],argv[4],argv[5],argv[6], true);
    }
    else
    if (strcmp(argv[2], "crc") == 0 && argc == 6)
    {
      return crcData(argv[3],argv[4],argv[5]);
    }
    else

    //mifare commands
    if (strcmp(argv[2], "wblock") == 0 && argc == 8)
    {
      return writeBlock(argv[3],argv[4],argv[5],argv[6],argv[7], false);
    }
    else
    if (strcmp(argv[2], "rblock") == 0 && argc == 8)
    {
      return readBlock(argv[3],argv[4],argv[5],argv[6],argv[7], false);
    }
    else
    if (strcmp(argv[2], "wdblock") == 0 && argc == 8)
    {
      return writeBlock(argv[3],argv[4],argv[5],argv[6],argv[7], true);
    }
    else
    if (strcmp(argv[2], "rdblock") == 0 && argc == 8)
    {
      return readBlock(argv[3],argv[4],argv[5],argv[6],argv[7], true);
    }
    else
    if (strcmp(argv[2], "rpage") == 0 && argc == 6)
    {
      return readPage(argv[3],argv[4],argv[5]);
    }
    else
    if (strcmp(argv[2], "wpage") == 0 && argc == 6)
    {
      return writePage(argv[3],argv[4],argv[5]);
    }
    else
    if (strcmp(argv[2], "rvalue") == 0 && argc == 7)
    {
      return readValue(argv[3],argv[4],argv[5],argv[6]);
    }
    else
    if (strcmp(argv[2], "wvalue") == 0 && argc == 8)
    {
      return writeValue(argv[3],argv[4],argv[5],argv[6],argv[7]);
    }
    else
    if (strcmp(argv[2], "incvalue") == 0 && argc == 7)
    {
      return incrementValue(argv[3],argv[4],argv[5],argv[6], false);
    }
    else
    if (strcmp(argv[2], "decvalue") == 0 && argc == 7)
    {
      return incrementValue(argv[3],argv[4],argv[5],argv[6], true);
    }
    else
    if (strcmp(argv[2], "transfervalue") == 0 && argc == 6)
    {
      return putgetValue(argv[3],argv[4],argv[5], B1RFIDCommandTransferValue);
    }
    else
    if (strcmp(argv[2], "restorevalue") == 0 && argc == 6)
    {
      return putgetValue(argv[3],argv[4],argv[5], B1RFIDCommandRestoreValue);
    }
    else
    if (strcmp(argv[2], "getversion") == 0)
    {
      return getVersion();
    }
    else
    if (strcmp(argv[2], "rsignature") == 0)
    {
      return readSignature();
    }
    else
    if (strcmp(argv[2], "cfguid") == 0 && argc == 6)
    {
      return cfgUid(argv[3],argv[4],argv[5]);
    }
    else
    if (strcmp(argv[2], "rcounter") == 0 && argc == 5)
    {
      return readCounter(argv[3],argv[4]);
    }
    else
    if (strcmp(argv[2], "icounter") == 0 && argc == 5)
    {
      return incrementCounter(argv[3],argv[4]);
    }
    else
    if (strcmp(argv[2], "checkevent") == 0 && argc == 5)
    {
      return checkEvent(argv[3],argv[4]);
    }
    else
    if (strcmp(argv[2], "passwdauth") == 0 && argc == 5)
    {
      return passwdAuth(argv[3],argv[4]);
    }
    else
    if (strcmp(argv[2], "polling") == 0 && argc == 17)
    {
      return pollingEnable(argv[3],argv[4],argv[5],argv[6],argv[7],argv[8],argv[9],argv[10],argv[11],argv[12],argv[13],argv[14],argv[15],argv[16]);
    }
    else
    if (strcmp(argv[2], "pwm") == 0 && argc == 6)
    {
      return enablePWM(argv[3],argv[4],argv[5]);
    }
    else
    if (strcmp(argv[2], "sgpio") == 0 && argc == 5)
    {
      return setGPIO(argv[3],argv[4]);
    }
    else
        print_usage();

    return -1;
}

static int setargs(char *args, char **argv)
{
    int count = 0;

    while (isspace(*args))
        ++args;
    while (*args)
    {
        if (argv)
            argv[count] = args;
        while (*args && !isspace(*args))
            ++args;
        if (argv && *args)
            *args++ = '\0';
        while (isspace(*args))
            ++args;
        count++;
    }
    return count;
}

char **parsedargs(char *args, int *argc)
{
    char **argv = NULL;
    int    argn = 0;

    if (args && *args
            && (args = strdup(args))
            && (argn = setargs(args,NULL))
            && (argv = malloc((argn+1) * sizeof(char *))))
    {
        *argv++ = args;
        argn = setargs(args,argv);
    }

    if (args && !argv)
        free(args);

    *argc = argn;
    return argv;
}

void freeparsedargs(char **argv)
{
    if (argv)
    {
        free(argv[-1]);
        free(argv-1);
    }
}

int tcpServerStart(int port)
{
    int listen_socket, client_socket, i, new_client, max_fd, res, addrlen;
    struct sockaddr_in addr;
    char buff[2000];
    fd_set rfds;

    addrlen = sizeof(addr);
    //Create socket
    listen_socket = socket(AF_INET , SOCK_STREAM , 0);
    if (listen_socket == -1)
    {
        printf("Could not create listen socket");
        return -1;
    }

    //Prepare the sockaddr_in structure
    addr.sin_family = AF_INET;
    addr.sin_addr.s_addr = INADDR_ANY;
    addr.sin_port = htons( port );


    printf("Starting TCP server on port %d.\n", port);

    if( bind(listen_socket,(struct sockaddr *)&addr , sizeof(addr)) < 0)
    {
        perror("bind failed. Error");
        return -1;
    }

    //max pending connection = 3
	if (listen(listen_socket, 3) < 0)
	{
		perror("listen");
		exit(EXIT_FAILURE);
	}

   client_socket = -1;

    while(1)
	{
		FD_ZERO(&rfds);
		FD_SET(listen_socket, &rfds);
		max_fd = listen_socket;

        if(client_socket >= 0)
            FD_SET( client_socket , &rfds);

        if(client_socket > max_fd)
			max_fd = client_socket;

		//wait for an activity on one of the sockets , timeout is NULL ,
		//so wait indefinitely
		res = select( max_fd + 1 , &rfds , NULL , NULL , NULL);

		if ((res < 0) && (errno!=EINTR))
		{
			printf("select error");
		}

		//If something happened on the master socket ,
		//then its an incoming connection
		if (FD_ISSET(listen_socket, &rfds))
		{
			if ((new_client = accept(listen_socket,
					(struct sockaddr *)&addr, (socklen_t*)&addrlen))<0)
			{
				perror("accept");
				exit(EXIT_FAILURE);
			}

			//inform user of socket number - used in send and receive commands
			printf("New connection , socket fd is %d , ip is : %s , port : %d\n" , new_client , inet_ntoa(addr.sin_addr) , ntohs(addr.sin_port));

            if( client_socket != -1)
            {
                close(client_socket);
                printf("Dropping old client connection %d\n" , i);
            }

            client_socket = new_client;
            printf("Adding to list of sockets as %d\n" , i);
            std_output_fd = client_socket;
		}

		//else its some IO operation on some other socket
        if (client_socket != -1 && FD_ISSET( client_socket , &rfds))
        {
            strcpy(buff,"rfidb1-tool device ");
            i = strlen(buff);
            if ((res = read(client_socket , &buff[i], sizeof(buff))) == 0)
            {
                printf("Host disconnected , ip %s , port %d \n" , inet_ntoa(addr.sin_addr) , ntohs(addr.sin_port));

                //Close the socket and mark as 0 in list for reuse
                close( client_socket );
                client_socket = -1;
                std_output_fd = 1;
            }
            else
            {
                char **av;
                int ac;
                buff[res+i] = '\0';
                av = parsedargs(buff,&ac);
                printf("Parsing %d args [%s]\n", ac, buff);
                if (ac > 2)
                {
                    if (strcmp(av[2], "poll") == 0)
                    {
                        own_printf("Command not allowed over TCP.");
                    }
                    else
                        parseCommands(ac, av);
                }
                freeparsedargs(av);
            }
        }

	}
    return 0;
}

int main(int argc, char * argv[])
{
    uint8_t rxBuff[MAX_FRAME_SIZE * 2];
    uint8_t txBuff[MAX_FRAME_SIZE * 2];
    uint16_t lenght;
    int optind, res;

    RFIDB1_InterfaceConfigurationT config;

    if (argc < 3)
      print_usage();

    serial_fd = OpenPort(argv[1], B1SerialBaudrate_Default);
    if (serial_fd == -1)
    {
        own_printf("Unable to open device %s for RFIDB1 module!\n", argv[1]);
        return -1;
    }

    GetRFIDB1Interface(&B1Interface);
    config.handleRequest = handleRequest;
    config.handleResponse = handleResponse;

    config.InputBuffer = rxBuff;
    config.InputBufferSize = sizeof(rxBuff);
    config.OutputBuffer = txBuff;
    config.OutputBufferSize = sizeof(txBuff);
    //config.AesDecryptBuffer = AES128_CBC_decrypt_buffer;
    //config.AesEncryptBuffer = AES128_CBC_encrypt_buffer;

    //initialise functions needed to access hardware, and system delay (optional, not needed for communication)
    config.getNSleep = getNSleep;
    config.setResetPin = setResetPin;
    config.setPowerPin = setPowerPin;
    config.delayMs = delayMs;

    if (B1Interface.Initialise(&B1Object, &config) != B1StatusT_OK)
    {
        own_printf("Unable to initialise RFIDB1_interface\n");
        return -1;
    }


    B1Interface.SetPacketHeaderType(&B1Object, HeaderTypeA);
    B1Interface.SetPacketEncoding(&B1Object, DataTypePlain, NULL, NULL);
    B1Interface.SetUserData(&B1Object, &serial_fd);

    if (argc == 4 && strcmp(argv[2], "server") == 0)
    {
        res = tcpServerStart(atoi(argv[3]));
    }
    else
        res = parseCommands(argc, argv);

    if (serial_fd != -1)
        close(serial_fd);

    return res;
}
